/* Copyright (c) 2016 Boulder Real Time Technologies, Inc. */
/* All rights reserved */

/* This software module is wholly owned by Boulder Real Time 
	Technologies, Inc. Any use of this software module without
	express written permission from Boulder Real Time Technologies,
	Inc. is prohibited. */

#ifndef __EV__
#define __EV__

#include <stddef.h>
#include <cstdarg>
#include <functional>
#include <string>
#include <vector>
#include <unordered_map>
#include <algorithm>
#include <memory>

#include "Array.h"
#include "db.h"
#include "stock.h"
#include "coords.h"
#include "BUConfigure.h"

#define EVOffset(type, field) ((unsigned long) ((char *) &((type *) 0)->field))

#define EVHasOnlyDigits(s) (s.find_first_not_of( "0123456789" ) == std::string::npos)

#define EVENTVIEW_MAGIC	(int)(0x1f3b2d56)

#define EVENTVIEW_TABLE_UNCHANGED	0
#define EVENTVIEW_TABLE_MODIFIED	1
#define EVENTVIEW_TABLE_BIGGER		2
#define EVENTVIEW_TABLE_SMALLER		3
#define EVENTVIEW_TABLE_NONE		4
#define EVENTVIEW_RETURN_RESET		10
#define EVENTVIEW_RETURN_BUSY		11

#define EVENTVIEW_ASSOCIATION_DEFINING_DEFINING		1
#define EVENTVIEW_ASSOCIATION_DEFINING_ASSOCIATED	0
#define EVENTVIEW_ASSOCIATION_DEFINING_OTHER		2

#define EVENTVIEW_TYPE_LONG			1
#define EVENTVIEW_TYPE_DOUBLE		2
#define EVENTVIEW_TYPE_STRINGPTR	3
#define EVENTVIEW_TYPE_EVEVENT_PTR	4
#define EVENTVIEW_TYPE_EVORIGIN_PTR	5
#define EVENTVIEW_TYPE_EVMAGNITUDE_PTR	6

#define EVENTVIEW_CALIB_FROM_DB_NO	0
#define EVENTVIEW_CALIB_FROM_DB_YES	1
#define EVENTVIEW_CALIB_FROM_DB_NULLONLY	2

#define EVENTVIEW_AUTO_SELECT_OFF			0
#define EVENTVIEW_AUTO_SELECT_MOSTRECENT	1
#define EVENTVIEW_AUTO_SELECT_LASTMODIFIED	2

class EVServer;
class EVClient;

struct EVCommand {
	EVCommand (std::string command) {
		memset (this, 0, EVOffset (EVCommand, command));
		this->command = command;
		recno = -1;
		id = -1;
	}

	double t1;
	double t2;
	int calib_from_db;
	long recno;
	long id;
	Tbl **kpattern;
	Tbl **tpattern;
	Hook **hook;
	EVClient *client;

	std::string command;
	std::string sta;
	std::string chan;
	std::string stachan;
	std::string stachan_reject;
	std::string seed_net;
	std::string seed_sta;
	std::string seed_chan;
	std::string seed_loc;
	std::string dbk_table;
	std::string dbk_record;
	std::string dbt_table;
};

struct EVResponse {
	EVResponse () {
		memset (this, 0, EVOffset (EVResponse, expanded_events_changed));
		last_modified_event_evid = -1;
		most_recent_event_evid = -1;
	}

	int database_reset;
	int events_changed;
	int detections_changed;
	int last_modified_event_changed;
	long last_modified_event_evid;
	int most_recent_event_changed;
	long most_recent_event_evid;
	Table<long> expanded_events_changed;
	Table<long> expanded_events_erased;
};

#define DFILETIME_BAD -9999999999.99900

struct EVTable {
	EVTable () {
		memset (this, 0, EVOffset (EVTable, name));
		dbtable = dbinvalid();
		mod_time = DFILETIME_BAD;
		mod_time_tables = DFILETIME_BAD;
	}

	long nrows;
	long nrows_tables;
	Dbptr dbtable;
	double mod_time;
	double mod_time_tables;
	int record_size;
	std::string name;
	std::string fname;
	std::string path;
	std::string record_null;
};

struct EVTableRecord {
	EVTableRecord () {
		recno = -1;
	}

	long recno;
	std::string record;
};

struct EVMatch {
	EVMatch () {
		memset (this, 0, sizeof(EVMatch));
	}
	~EVMatch () {
		// if (kpat) freetbl (kpat, free);
		// if (tpat) freetbl (tpat, free);
// if (hook) free_hook (&hook);
	}

	Tbl *kpat;
	Tbl *tpat;
	Hook *hook;
};

struct EVStaChan {
	EVStaChan () {
		memset (this, 0, sizeof(EVStaChan));
	}

	char sta[32];
	char chan[32];
	double time;
	double endtime;
};

struct EVArrival {
	EVArrival () {
		arid = -1;
		recno_arrival = -1;
		lddate_arrival = 0.0;
		link_to_any_assoc = 0;
		link_to_any_stamag = 0;
		link_to_any_wfmeas = 0;
	}

	long arid;
	long recno_arrival;
	char record_arrival[274];
	double lddate_arrival;

	int link_to_any_assoc;
	int link_to_any_stamag;
	int link_to_any_wfmeas;

	Table<long> associated_orids;
};

struct EVWfmeas {
	EVWfmeas () {
		memset (this, 0, sizeof(EVWfmeas));
		recno_wfmeas = -1;
	}

	long recno_wfmeas;
	char record_wfmeas[256];
	double lddate_wfmeas;
};

struct EVAssociation {
	EVAssociation () {
		memset (this, 0, sizeof(EVAssociation));
		recno_assoc = -1;
		recno_arrival = -1;
		recno_predarr = -1;
		recno_site = -1;
		arid = -1;
	}

	long recno_assoc;
	char record_assoc[190];
	double lddate_assoc;

	long recno_arrival;
	char record_arrival[274];
	double lddate_arrival;
	double time_arrival;

	long recno_predarr;
	char record_predarr[104];

	long recno_site;
	char record_site[202];

	long arid;
	int defining;
};

struct EVDetection {
	EVDetection () {
		memset (this, 0, sizeof(EVDetection));
		recno_detection = -1;
		recno_site = -1;
		presidual = -999.0;
		sresidual = -999.0;
		delta = -999.0;
	}

	long recno_detection;
	char record_detection[204];
	double lddate_detection;

	long recno_site;
	char record_site[202];

	double presidual;
	double sresidual;
	double delta;
};

struct EVMoment {
	EVMoment () {
		memset (this, 0, sizeof(EVMoment));
		recno_mt = -1;
	}

	long recno_mt;
	char record_mt[564];
	double lddate_mt;
};

struct EVPreferredMagnitude {
	EVPreferredMagnitude () {
		memset (this, 0, sizeof(EVPreferredMagnitude));
	}

	char magtype[8];
	double minimum_magnitude;
	double maximum_magnitude;
};

struct EVStamag {
	EVStamag () {
		memset (this, 0, EVOffset (EVStamag, wfmeass));
		recno_stamag = -1;
		arid = -1;
		magnitude = -999.0;
	}

	long recno_stamag;
	char record_stamag[168];
	double lddate_stamag;

	long arid;
	double magnitude;
	char magtype[8];

	Table<EVWfmeas> wfmeass;
};

struct EVMagnitude {
	EVMagnitude () {
		memset (this, 0, EVOffset (EVMagnitude, stamags));
		recno_netmag = -1;
		magnitude = -999.0;
	}

	long recno_netmag;
	char record_netmag[144];
	double lddate_netmag;

	double magnitude;
	char magtype[8];

	Table<EVStamag> stamags;
};

struct EVOrigin {
	EVOrigin () {
		memset (this, 0, EVOffset (EVOrigin, magnitudes));
		recno_origin = -1;
		orid = -1;
		magnitude = -999.0;
		recno_origerr = -1;
		recno_emodel = -1;
		pref_magnitude = -1;
		pref_moment = -1;
	}

	long recno_origin;
	char record_origin[288];
	long orid;
	double lddate_origin;
	double lddate_origin_view;
	double time;

	long recno_origerr;
	char record_origerr[268];

	long recno_emodel;
	char record_emodel[96];

	double magnitude;
	char magtype[8];

	int preferred;
	int pref_magnitude;
	int pref_moment;

	Table<EVMagnitude> magnitudes;
	Table<EVMoment> moments;
	Table<EVDetection> detections;
	Table<EVAssociation> associations;
};

struct EVEvent {
	EVEvent () {
		memset (this, 0, EVOffset (EVEvent, origins));
		recno_event = -1;
		evid = -1;
		prefor = -1;
		order = -1;
		magnitude = -999.0;
		pref_origin = -1;
		pref_magnitude_origin = -1;
		pref_moment_origin = -1;
	}

	long recno_event;
	char record_event[154];
	long evid;
	long prefor;
	long order;
	double lddate_event;
	double lddate_view;
	double access_time;
	double prefor_time;

	double magnitude;
	char magtype[8];

	int pref_origin;
	int pref_magnitude_origin;
	int pref_moment_origin;

	Table<EVOrigin> origins;
};

struct EVOriginsTab {
	EVOriginsTab () {
		origins = NULL;
		row_to_index_tab.clear();
		index_to_row_tab.clear();
		row = -1;
	}
	EVOriginsTab (Table<EVOrigin> *origins) {
		this->origins = origins;
		row = -1;
		row_to_index_tab.clear();
		index_to_row_tab.clear();
		if (origins) {
			row_to_index_tab = Table<int>(origins->size());
			index_to_row_tab = Table<int>(origins->size());
			for (int i=0; i<origins->size(); i++) {
				row_to_index_tab.push_back (i);
				index_to_row_tab.push_back (i);
			}
		}
	}

	Table<EVOrigin> *origins;
	Table<int> row_to_index_tab;
	Table<int> index_to_row_tab;
	int row;
};

struct EVServerClient {
	EVServerClient () {
		this->client = NULL;
	}

	EVClient *client;
};

struct EVVars {
	EVVars () {
		magic=EVENTVIEW_MAGIC;
		auto_refresh = 0.0;
		auto_check_database = 0.0;
		process_detections = 1;
		age_ref_now = 1;
		max_events = -1;
		check_nfs = 1;
		ret = 0;
		cmd_mtfifo = NULL;
		run_thread = (pthread_t) NULL;
		refresh_events = (pthread_t) NULL;
		database_changed = (pthread_t) NULL;
		dblock = PTHREAD_MUTEX_INITIALIZER;
		eventslock = PTHREAD_MUTEX_INITIALIZER;
		expandedeventslock = PTHREAD_MUTEX_INITIALIZER;
		auto_refresh_mutex = PTHREAD_MUTEX_INITIALIZER;
		scratch_mutex = PTHREAD_MUTEX_INITIALIZER;
		db = dbinvalid();
		evid.kpat = strtbl ((char *)"evid", NULL);
		evid.tpat = strtbl ((char *)"evid", NULL);
		orid.kpat = strtbl ((char *)"orid", NULL);
		orid.tpat = strtbl ((char *)"orid", NULL);
		event_origin.kpat = strtbl ((char *)"evid", NULL);
		event_origin.tpat = strtbl ((char *)"evid", NULL);
		origin_origerr.kpat = strtbl ((char *)"orid", NULL);
		origin_origerr.tpat = strtbl ((char *)"orid", NULL);
		origin_emodel.kpat = strtbl ((char *)"orid", NULL);
		origin_emodel.tpat = strtbl ((char *)"orid", NULL);
		origin_netmag.kpat = strtbl ((char *)"orid", NULL);
		origin_netmag.tpat = strtbl ((char *)"orid", NULL);
		origin_mt.kpat = strtbl ((char *)"orid", NULL);
		origin_mt.tpat = strtbl ((char *)"orid", NULL);
		origin_assoc.kpat = strtbl ((char *)"orid", NULL);
		origin_assoc.tpat = strtbl ((char *)"orid", NULL);
		assoc_arrival.kpat = strtbl ((char *)"arid", NULL);
		assoc_arrival.tpat = strtbl ((char *)"arid", NULL);
		assoc_predarr.kpat = strtbl ((char *)"arid", (char *)"orid", NULL);
		assoc_predarr.tpat = strtbl ((char *)"arid", (char *)"orid", NULL);
		arrival_assoc.kpat = strtbl ((char *)"arid", NULL);
		arrival_assoc.tpat = strtbl ((char *)"arid", NULL);
		arrival_stamag.kpat = strtbl ((char *)"arid", NULL);
		arrival_stamag.tpat = strtbl ((char *)"arid", NULL);
		arrival_wfmeas.kpat = strtbl ((char *)"arid", NULL);
		arrival_wfmeas.tpat = strtbl ((char *)"arid", NULL);
		stamag_wfmeas.kpat = strtbl ((char *)"arid", NULL);
		stamag_wfmeas.tpat = strtbl ((char *)"arid", NULL);
		netmag_stamag.kpat = strtbl ((char *)"magid", NULL);
		netmag_stamag.tpat = strtbl ((char *)"magid", NULL);
		origin_detection.kpat = strtbl ((char *)"time-10.0::time+1800.0", NULL);
		origin_detection.tpat = strtbl ((char *)"time", NULL);
		detection.tpat = strtbl ((char *)"time", NULL);
		arrival.kpat = strtbl ((char *)"sta", (char *)"chan", (char *)"time::endtime", NULL);
		arrival.tpat = strtbl ((char *)"sta", (char *)"chan", (char *)"time", NULL);
		rawdetection.kpat = strtbl ((char *)"sta", (char *)"chan", (char *)"time::endtime", NULL);
		rawdetection.tpat = strtbl ((char *)"sta", (char *)"chan", (char *)"time", NULL);
		wfdisc.kpat = strtbl ((char *)"sta", (char *)"chan", (char *)"time::endtime", NULL);
		wfdisc.tpat = strtbl ((char *)"sta", (char *)"chan", (char *)"time::endtime", NULL);
		calibration.kpat = strtbl ((char *)"sta", (char *)"chan", (char *)"time::endtime", NULL);
		calibration.tpat = strtbl ((char *)"sta", (char *)"chan", (char *)"time::endtime", NULL);
		tables["event"] = EVTable();
		tables["origin"] = EVTable();
		tables["origerr"] = EVTable();
		tables["emodel"] = EVTable();
		tables["assoc"] = EVTable();
		tables["arrival"] = EVTable();
		tables["predarr"] = EVTable();
		tables["wfmeas"] = EVTable();
		tables["netmag"] = EVTable();
		tables["stamag"] = EVTable();
		tables["mt"] = EVTable();
		tables["site"] = EVTable();
		tables["detection"] = EVTable();
		tables["wfdisc"] = EVTable();
		tables["calibration"] = EVTable();
		maxnumber_expanded_events = 10;
	}
	~EVVars () {
		if (db.database != dbINVALID) dbclose (db);
	}

	int magic;

	double auto_refresh;
	double auto_check_database;
	int process_detections;
	int age_ref_now;
	int max_events;
	int check_nfs;
	int ret;
	Pmtfifo *cmd_mtfifo;
	pthread_t run_thread;
	pthread_t refresh_events;
	pthread_t database_changed;
	pthread_mutex_t dblock;
	pthread_mutex_t eventslock;
	pthread_mutex_t expandedeventslock;
	pthread_mutex_t auto_refresh_mutex;
	pthread_mutex_t scratch_mutex;

	EVMatch evid;
	EVMatch orid;
	EVMatch event_origin;
	EVMatch origin_origerr;
	EVMatch origin_emodel;
	EVMatch origin_netmag;
	EVMatch origin_mt;
	EVMatch origin_assoc;
	EVMatch assoc_arrival;
	EVMatch assoc_predarr;
	EVMatch arrival_assoc;
	EVMatch arrival_stamag;
	EVMatch arrival_wfmeas;
	EVMatch stamag_wfmeas;
	EVMatch site;
	EVMatch netmag_stamag;
	EVMatch origin_detection;
	EVMatch detection;
	EVMatch arrival;
	EVMatch rawdetection;
	EVMatch wfdisc;
	EVMatch calibration;

	std::string dbname;
	std::string schema;
	Dbptr db;
	Array<std::string,EVTable> tables;

	Table<EVServerClient> clients;

	Table<EVPreferredMagnitude> preferred_magnitudes;

	int maxnumber_expanded_events;
};

class EVServer {
	friend class EVClient;
	
public:
	EVServer (std::string dbname, int age_ref_now = 1, int max_events = -1, int check_nfs = 1);
	~EVServer();

	int refreshEvents (int force=0);
	void autoRefreshEvents (double interval);
	void autoCheckDatabase (double interval);
	void setAgeRefNow (int value);
	void setMaxEvents (int max_events);
	void printDifferences (Table<EVEvent> *events);
	int setPreferredMagnitudes (char *pf_string);
	int setPreferredMagnitudes (Pf *pf);
	std::string getDbname ();
	std::string getSchema ();
	int reset (int lockdb=0);

	static int findEvid (Table<EVEvent> *events, long evid);
	static long getSize (Table<EVEvent> *events);
	static long getSize (EVEvent *event);
	static long getSize (Table<EVDetection> *detections);
	static long getSize (EVDetection *detection);
	static long getSize (EVAssociation *association);
	static long getSize (EVOrigin *origin);
	static long getSize (EVMagnitude *magnitude);
	static long getSize (EVStamag *stamag);
	static long getSize (EVMoment *moment);
	static long getSize (EVWfmeas *wfmeas);
	static int isEqual (EVEvent *event1, EVEvent *event2);
	static int isEqual (EVOrigin *origin1, EVOrigin *origin2);
	static int isEqual (EVMagnitude *magnitude1, EVMagnitude *magnitude2);
	static int isEqual (EVStamag *stamag1, EVStamag *stamag2);
	static int isEqual (EVMoment *moment1, EVMoment *moment2);
	static int isEqual (EVWfmeas *wfmeas1, EVWfmeas *wfmeas2);
	static int isEqual (EVAssociation *association1, EVAssociation *association2);
	static int isEqual (EVDetection *detection1, EVDetection *detection2);
	static void print (Table<EVEvent> *events, std::string pre="");
	static void print (Table<EVOrigin> *origins, std::string pre="");
	static void print (Table<EVMagnitude> *magnitudes, std::string pre="");
	static void print (Table<EVStamag> *stamags, std::string pre="");
	static void print (Table<EVMoment> *moments, std::string pre="");
	static void print (Table<EVWfmeas> *wfmeass, std::string pre="");
	static void print (Table<EVAssociation> *associations, std::string pre="");
	static void print (Table<EVDetection> *detections, std::string pre="");
	static void print (Table<EVTableRecord> *records, std::string pre="");
	static void print (Table<EVStaChan> *stachans, std::string pre="");
	static void print (EVEvent *event, std::string pre="");
	static void print (EVOrigin *origin, std::string pre="");
	static void print (EVMagnitude *magnitude, std::string pre="");
	static void print (EVStamag *stamag, std::string pre="");
	static void print (EVMoment *moment, std::string pre="");
	static void print (EVWfmeas *wfmeas, std::string pre="");
	static void print (EVDetection *detection, std::string pre="");
	static void print (EVAssociation *association, std::string pre="");
	static void print (EVTableRecord *record, std::string pre="");
	static void print (EVStaChan *stachan, std::string pre="");
	static void print (Table<EVEvent> *events1, Table<EVEvent> *events2, std::string pre="");
	static void print (Table<EVOrigin> *origins1, Table<EVOrigin> *origins2, std::string pre="");
	static void print (Table<EVMagnitude> *magnitudes1, Table<EVMagnitude> *magnitudes2, std::string pre="");
	static void print (Table<EVStamag> *stamags1, Table<EVStamag> *stamags2, std::string pre="");
	static void print (Table<EVMoment> *moments1, Table<EVMoment> *moments2, std::string pre="");
	static void print (Table<EVWfmeas> *wfmeass1, Table<EVWfmeas> *wfmeass2, std::string pre="");
	static void print (Table<EVAssociation> *associations1, Table<EVAssociation> *associations2, std::string pre="");
	static void print (Table<EVDetection> *detections1, Table<EVDetection> *detections2, std::string pre="");
	static void print (EVEvent *event1, EVEvent *event2, std::string pre="");
	static void print (EVOrigin *origin1, EVOrigin *origin2, std::string pre="");
	static void print (EVMagnitude *magnitude1, EVMagnitude *magnitude2, std::string pre="");
	static void print (EVStamag *stamag1, EVStamag *stamag2, std::string pre="");
	static void print (EVMoment *moment1, EVMoment *moment2, std::string pre="");
	static void print (EVWfmeas *wfmeas1, EVWfmeas *wfmeas2, std::string pre="");
	static void print (EVAssociation *association1, EVAssociation *association2, std::string pre="");
	static void print (EVDetection *detection1, EVDetection *detection2, std::string pre="");

	Table<EVEvent> events;
	double events_last_time = 0.0;
	Table<EVDetection> detections;

	Array<long, EVEvent> expanded_events;
	long last_modified_evid = -1;
	long most_recent_evid = -1;
	
	int stop=0;

private:
	int launchThread ();
	int eventsNumber ();
	long getEventsSize ();
	int detectionsNumber ();
	long getDetectionsSize ();
	int expandedNumber ();
	long getExpandedSize ();
	int getEvent (int index, EVEvent &event);
	int getEvent (double tstart, double tend, EVEvent &event);
	int getEventFromOrid (long orid, EVEvent &event);
	int getEvid (long evid, EVEvent &event);
	int findEvid (long evid);
	int getLastModifiedEvent (EVEvent &event);
	long getLastModifiedEvid ();
	int getMostRecentEvent (EVEvent &event);
	long getMostRecentEvid ();
	Table<EVEvent>::iterator getEventsIterator ();
	int isEqual (Table<EVEvent> *events);
	int isEqual (Table<EVDetection> *detections);

	int setDb (std::string dbname);
	void closeDb ();
	void freeHooks ();
	void freeVectors ();
	int checkTable (std::string name, int replace=1, int verbose=0);
	int checkTable (EVTable *table, int replace=1, int verbose=0);
	int openTable (Dbptr db, std::string name, int check_remote=1);
	int checkTableTables (std::string name, int replace=1);
	int checkTableTables (EVTable *table, int replace=1);
	int selectEvent (int index);
	int refreshEvent (long dbevent_record, EVEvent *event, long *order);
	int refreshOrigin (long dborigin_record, EVOrigin *origin);
	int refreshDetections (double t1, double t2, Table<EVDetection> *detections);
	int refreshDetections (EVOrigin *origin, int all);
	int refreshDetections (EVEvent *event);
	int refreshAssociations (EVEvent *event, int refresh_detections);
	int refreshAssociations (EVOrigin *origin);
	int refreshAssociation (EVOrigin *origin, long dbassoc_record, EVAssociation *association);
	int refreshMagnitudes (EVOrigin *origin);
	int refreshStamags (EVMagnitude *magnitude);
	int refreshStamag (long dbstamag_record, EVStamag *stamag);
	int containsExpandedEvent (Array<long, EVEvent> expanded_events, long evid, EVEvent &event);
	int containsExpandedEvent (Array<long, EVEvent> expanded_events, long evid);
	int containsExpandedEvent (long evid, EVEvent &event);
	int containsExpandedEvent (long evid);
	void addExpandedEvent (EVEvent event);
	void purgeExpandedEvents ();
	void grabEventsLock ();
	void releaseEventsLock ();
	void registerClient (EVClient *client);
	void deregisterClient (EVClient *client);
	int getExpandedEvent (EVClient *client, long evid, EVEvent &event);
	static void purgeExpandedEvents (int maxnumber_expanded_events, Array<long, EVEvent> *expanded_events);
	Table<EVEvent> getEvents ();
	Table<EVDetection> getDetections ();
	Table<EVArrival> getArrivals (std::string sta, std::string chan, double t1, double t2);
	Table<EVTableRecord> getWfdiscs (std::string sta, std::string chan, double t1, double t2, int calib_from_db);
	Table<EVDetection> getDetections (std::string sta, std::string chan, double t1, double t2);
	Table<EVTableRecord> getRawDetections (std::string sta, std::string chan, double t1, double t2);
	EVTableRecord getSite (std::string sta, double time);
	Table<EVStaChan> scanWfdisc (std::string stachan_accept_expr=std::string(),
								std::string stachan_reject_expr=std::string(),
								double t1=-1.e30, double t2=1.e30);
	void scanArrival (Table<EVStaChan> &associated, Table<EVStaChan> &unassociated,
							std::string sta_expr=std::string(), std::string chan_expr=std::string(), 
							double t1=-1.e30, double t2=1.e30);
	void checkDatabase();
	Table<EVTableRecord> execDbmatches (std::string dbk_table, std::string dbk_record, 
									std::string dbp_table, Tbl **kpattern, Tbl **dpattern, Hook **hook);
	Table<EVTableRecord> execDbsubset (std::string table_name, std::string subset_expr);
	long dbputrecord (std::string table_name, long recno, long id, std::string record);
	long execDbnextid (std::string id_name);
	long execDbseed2css (std::string seed_net, std::string seed_sta, 
										std::string seed_chan, std::string seed_loc,
										std::string &css_sta, std::string &css_chan);
	long execDbcss2seed (std::string css_sta, std::string css_chan,
						std::string &seed_net, std::string &seed_sta, std::string &seed_chan, std::string &seed_loc);
	int getsetScratch (EVTable *table, long recno, Dbptr *dbptr, char *scratch_record);
	int getsetScratch (EVTable *table, long recno, Dbptr *dbptr, EVTableRecord *scratch_record);

	static void *run_thread (void *arg);
	static void *refresh_events_thread (void *arg);
	static void *database_changed_thread (void *arg);
	static int getPreferredMagnitudePriority (Table<EVPreferredMagnitude> *preferred_magnitudes, EVMagnitude *magnitude);
	static int findPreferredMagnitude (Table<EVPreferredMagnitude> *preferred_magnitudes, Table<EVMagnitude> *magnitudes, int *priority=NULL);
	static int readPreferredMagnitudes (char *pf_string, Table<EVPreferredMagnitude> &preferred_magnitudes);
	static int readPreferredMagnitudes (Pf *pf, Table<EVPreferredMagnitude> &preferred_magnitudes);
	static void setPreferredMagnitude (Table<EVPreferredMagnitude> *preferred_magnitudes, EVOrigin *origin);
	static void setPreferredMagnitude (Table<EVPreferredMagnitude> *preferred_magnitudes, EVEvent *event);

	EVVars vars;
};

typedef struct EVQueryToken {
	std::string name;
	std::string index_key;
	std::string index_val;
} EVQueryToken;

struct EVEventsTab;

class EVClient {
	friend class EVServer;

public:
	EVClient (EVServer *ev);

	void refreshEvents (int wait_flag=0);
	void refreshEventsForce (int wait_flag=0);
	void grabEventsLock ();
	void releaseEventsLock ();
	int eventsNumber ();
	long getEventsSize ();
	int detectionsNumber ();
	long getDetectionsSize ();
	int expandedNumber ();
	long getExpandedSize ();
	EVOrigin *findOrid (EVEvent *event, long orid);
	std::shared_ptr<Table<EVEvent>> getEvents ();
	Table<EVDetection> getDetections ();
	int getEvent (int index, EVEvent &event);
	int getEvent (long evid, EVEvent &event);
	int getEvent (double tstart, double tend, EVEvent &event);
	int getEventFromOrid (long orid, EVEvent &event);
	int containsExpandedEvent (long evid);
	int getExpandedEvent (long evid, EVEvent &event);
	int getLastModifiedEvent (EVEvent &event);
	long getLastModifiedEvid ();
	int getMostRecentEvent (EVEvent &event);
	long getMostRecentEvid ();
	Table<EVArrival> getArrivals (std::string sta, std::string chan, double t1, double t2);
	std::string getWfdiscPath ();
	Table<EVTableRecord> getWfdiscs (std::string sta, std::string chan, double t1, double t2, int calib_from_db);
	Table<EVTableRecord> getRawDetections (std::string sta, std::string chan, double t1, double t2);
	EVTableRecord getSite (std::string sta, double time);
	Table<EVStaChan> scanWfdisc (std::string stachan_accept_expr=std::string(),
									std::string stachan_reject_expr=std::string(),
									double t1=-1.e30, double t2=1.e30);
	Table<EVTableRecord> dbmatches (std::string dbk_table, std::string dbk_record, 
									std::string dbp_table, Tbl **kpattern, Tbl **dpattern, Hook **hook);
	Table<EVTableRecord> dbsubset (std::string table_name, std::string subset_expr);
	long dbputrecord (std::string table_name, long recno, long id, std::string record);
	long dbnextid (std::string id_name);
	int dbcss2seed (std::string css_sta, std::string css_chan, 
			std::string &seed_net, std::string &seed_sta, std::string &seed_chan, std::string &seed_loc);
	int dbseed2css (std::string seed_net, std::string seed_sta, std::string seed_chan, std::string seed_loc,
			std::string &css_sta, std::string &css_chan); 
	void checkDatabase ();
	int getDbptr (EVEvent *event, Dbptr *dbptr_event);
	int getDbptr (EVOrigin *origin, Dbptr *dbptr_origin, Dbptr *dbptr_origerr=NULL, Dbptr *dbptr_emodel=NULL);
	int getDbptr (EVMagnitude *magnitude, Dbptr *dbptr_netmag);
	int getDbptr (EVMoment *moment, Dbptr *dbptr_mt);
	int getDbptr (EVAssociation *association, Dbptr *dbptr_assoc, Dbptr *dbptr_arrival=NULL, Dbptr *dbptr_site=NULL, Dbptr *dbptr_predarr=NULL);
	int getDbptr (EVDetection *detection, Dbptr *dbptr_detection, Dbptr *dbptr_site=NULL);
	int getDbptr (EVArrival *arrival, Dbptr *dbptr_arrival);
	int getDbptr (std::string table_name, char *record, Dbptr *dbptr);
	int getDbptr (std::string table_name, EVTableRecord *record, Dbptr *dbptr);
	int getDbptr (std::string table_name, Dbptr *dbptr);
	void printDifferences (Table<EVEvent> *events);
	int getValue (Table<EVEvent> *events, std::string query, int *type, void *value);
	int getValue (Table<EVOrigin> *origins, std::string query, int *type, void *value);
	std::string getDbname ();
	std::string getSchema ();
	int checkSortQuery (Table<EVEvent> *events, std::string query);
	int checkSortQuery (Table<EVOrigin> *origins, std::string query);
	void sort (EVEventsTab *view, std::string query, int reverse);
	void sort (EVOriginsTab *view, std::string query, int reverse);
	Table<int> sort (Table<EVEvent> *events, std::string query, int reverse, int *ret=NULL);
	Table<int> sort (Table<EVOrigin> *origins, std::string query, int reverse, int *ret=NULL);

	static EVOrigin *getPreferredOrigin (EVEvent *event, int *index=NULL);
	static EVMoment *getPreferredMoment (EVEvent *event);
	static EVMoment *getPreferredMoment (EVOrigin *origin);
	static int getOrigin (EVEvent *event, long orid, EVOrigin &origin, int return_preferred=1);
	static int getEvid (Table<EVEvent> *events, long evid);
	static int getOrid (Table<EVOrigin> *origins, long evid);

	static void setEventsTabs(EVEventsTab *view, Table<int> *sort_tab=NULL);
	static int getRow (EVEventsTab *view, int index);
	static int getIndex (EVEventsTab *view, int row);
	static int findEvidRow (EVEventsTab *view, long evid);
	static int findEvidIndex (EVEventsTab *view, long evid);
	static void setOriginsTabs(EVOriginsTab *view, Table<int> *sort_tab=NULL);
	static int getRow (EVOriginsTab *view, int index);
	static int getIndex (EVOriginsTab *view, int row);
	static int findOridRow (EVOriginsTab *view, long orid);
	static int findOridIndex (EVOriginsTab *view, long orid);
	static int getValueFromDbptr (Dbptr db, std::string expr, Expression **expr_compiled, int *dbtype_ret, int *type, void *value);

	int waitfor_events = 0;
	int waitfor = 0;
	
	EVServer *ev;

protected:
	virtual void responseCallback (EVResponse response) {};
	virtual void checkDatabaseCallback (Array<std::string, int> changes) {};

private:
	static int parse_query_token (std::string &query, EVQueryToken &token);
	static int parse_query (std::string query, std::vector<EVQueryToken> &tokens);
	int processValueDb (char *record, std::string table_name, std::string name, int *type, void *value);
	int getValueDb (char *record, std::string name, int *type, void *value);
	int getValue (Table<EVEvent> *events, std::vector<EVQueryToken> tokens, int *type, void *value);
	int getValue (Table<EVOrigin> *origins, std::vector<EVQueryToken> tokens, int *type, void *value);
	int getValue (EVEvent *event, std::vector<EVQueryToken> tokens, int *type, void *value);
	int getValue (EVOrigin *origin, std::vector<EVQueryToken> tokens, int *type, void *value);

	Dbptr db;
	Array<std::string, Expression *> expressions_arr;
	Array<std::string, int> dbtype_arr;
	Table<EVTableRecord> records;
	Table<EVArrival> arrivals;
	EVTableRecord record;
	Table<EVStaChan> stachans;
	std::string net;
	std::string sta;
	std::string chan;
	std::string loc;
	long return_value;
	long id;
};

struct EVEventsTab {
	EVEventsTab () {
		events = NULL;
		row_to_index_tab.clear();
		index_to_row_tab.clear();
		row = -1;
	}
	EVEventsTab (EVClient *client) {
		events_shared = std::make_shared<Table<EVEvent>>();
		events = events_shared.get();
		*events = client->ev->events;		
		row = -1;
		row_to_index_tab.clear();
		index_to_row_tab.clear();
		if (events) {
			row_to_index_tab = Table<int>(events->size());
			index_to_row_tab = Table<int>(events->size());
			for (int i=0; i<events->size(); i++) {
				row_to_index_tab.push_back (i);
				index_to_row_tab.push_back (i);
			}
		}
	}
	EVEventsTab (std::shared_ptr<Table<EVEvent>> ptr) {
		events_shared = ptr;
		events = events_shared.get();
		row = -1;
		row_to_index_tab.clear();
		index_to_row_tab.clear();
		if (events) {
			row_to_index_tab = Table<int>(events->size());
			index_to_row_tab = Table<int>(events->size());
			for (int i=0; i<events->size(); i++) {
				row_to_index_tab.push_back (i);
				index_to_row_tab.push_back (i);
			}
		}
	}

	std::shared_ptr<Table<EVEvent>> events_shared;
	Table<EVEvent> *events;
	Table<int> row_to_index_tab;
	Table<int> index_to_row_tab;
	int row;
};

#endif