/* Copyright (c) 2000 Boulder Real Time Technologies, Inc. */
/* All rights reserved */

#ifndef __brttutil__
#define __brttutil__

/* This software module is wholly owned by Boulder Real Time 
   Technologies, Inc. Any use of this software module without
   express written permission from Boulder Real Time Technologies,
   Inc. is prohibited. */

#include <pthread.h>
#include <pthread.h>

#include "db.h"
#include "response.h"
#include "stock.h"
#include "Pkt.h"

#ifdef  __cplusplus
extern "C" {
#endif

/* defines for parse_param */
#define	P_SINT	0	/* returns short int */
#define	P_LINT	1	/* returns long int */
#define	P_FLT	2	/* returns float */
#define	P_DBL	3	/* returns double */
#define	P_STR	4	/* returns string */
#define	P_PFTBL	5	/* returns Tbl with pf entries */
#define	P_TBL	6	/* returns Tbl with string entries */
#define	P_TBLPF	7	/* returns pf corresponding to a table input */
#define	P_PFARR	8	/* returns Arr with pf entries */
#define	P_ARR	9	/* returns Arr with string entries */
#define	P_ARRPF	10	/* returns pf corresponding to an array input */
#define	P_BOOL	11	/* returns boolean (int) */
#define	P_STRCP	12	/* copies into string */

/* return codes for pmtfifo_pop */
#define	PMTFIFO_ERROR		(-1)	/* error return */
#define	PMTFIFO_RELEASED	(0)	/* no data available and block released*/
#define	PMTFIFO_OK		(1)	/* data returned */
#define	PMTFIFO_TIMEOUT		(2)	/* timeout with no data available */
#define	PMTFIFO_NODATA		(3)	/* no data available on non-blocking FIFO */
#define	PMTFIFO_EOF		(4)	/* no more data available for managed fifos */

typedef struct pmtfifoelement_ {
	void *ptr;
	struct pmtfifoelement_ *next;
	struct pmtfifoelement_ *prev;
} PmtfifoElement;

typedef struct pmtfifo_ {
	pthread_mutex_t mutex;
	pthread_cond_t queue_condition;
	pthread_cond_t block_condition;
	int maxqueue;
	int queue;
	int flush;
	int block;
	int check;
	int release_block;
	double timeout;
	PmtfifoElement *head;
	PmtfifoElement *tail;
} Pmtfifo;

typedef struct pmtmanagedfifo_data_ {
	Hook *data;
	unsigned long sequence;
	unsigned long refcount;
	Hook * (*copyproc) (Hook *);
} PmtmanagedfifoData;

typedef struct pmtmanagedfifo_ {
	char name[64];
	Tbl *consumers_tbl;
	Arr *consumers_arr;
	Arr *producers_arr;
	pthread_mutex_t mutex;
	pthread_mutex_t mutex_write;
	unsigned long sequence;
} Pmtmanagedfifo;

typedef struct pmtcomp_ {
	pthread_mutex_t mutex;
	Tbl *threads;
	long quit;
} Pmtcomp;

typedef struct pmtcompthread_ {
	pthread_t thread;
	pthread_mutex_t mutex;
	pthread_cond_t start_cond;
	pthread_cond_t finished_cond;
	long start_ready;
	long finished_ready;
	long quit;
	long index;
	long ret;
	long (*comp_sub) (long ioffset, long number, void *user_data);
	long ioffset;
	long number;
	void *user_data;
} PmtcompThread;

typedef struct pf2struc_def_ {
	char *name;
	int pftype;
	int required;
	int struc_offset;
} Pf2strucDef;

/* return codes for buvector */
#define	BUVECTOR_NULL_DOUBLE		(3.4e38)	/* flagged NULL value for double */

typedef struct buvector_ {
	char ident[8];
	int ny;
	int nx;
	int size;
	char *tag;
	struct point_ {
		double x;
		double *y;
		char *label;
	} *point;
	double time;
	double dt;
	double *data;
	int heap_size;
} BUVector;

typedef struct buvector2_ {
	char ident[8];
	int ny;
	int nx;
	char tag[64];
	double time;
	double dt;
	int is_timeseries;
	int x_size;
	int y_size;
	int dblbuf_size;
	double *dblbuf;
	int lblbuf_size;
	char **lblbuf;
	int heap_size;
} BUVector2;

typedef struct thpoint_ {
    double time ; 
    double value ; 
} BUHistoryPoint ;

typedef struct thistory_ {
	int maxnp;
	char *tag;
	char *filename ;
	Tbl *points ;
} BUHistory;

typedef struct budsseis_ {
	Dbptr dbin;
	Dbptr dbscgr;
	Dbptr dbsc;
} BUDsseis;

#define	BUGRID_PARAMS_SIZE	20

typedef struct bugrid_ {
	char *tag;
	char *grid_type;
	int nxgrid;
	float *xgrid_coords;
	char *xgrid_type;
	char *xgrid_units;
	int nygrid;
	float *ygrid_coords;
	char *ygrid_type;
	char *ygrid_units;
	int nsamps;
	double time;
	double dt;
	float *data;
	float gap_value;
	int data_size;
	char *data_type;
	char *data_units;
	int order;
	double params[BUGRID_PARAMS_SIZE];
} BUGrid;

typedef struct buspec_ {
	char *spec_type;
	char *am_units;
	double twin;
	int nf;
	float *f;
	float *am;
	float *ph;
	size_t size;
} BUSpec;

#define BUSAC_TYPE_DOUBLE	1
#define BUSAC_TYPE_LONG		2
#define BUSAC_TYPE_STRING	3

typedef struct busacvalue_ {
	char *name;
	int type;
	union {
		int i;
		long int l;
		float f;
		double d;
		char *s;
	} value;
} BUSacValue;

typedef struct busac_ {
	char *tag;
	void *sacimage;
} BUSac;

#define	BUSCV_ROWS_PER_THREAD		10000

#define	BUSCV_READ_FLAGS_WFDISC_ONLY	0
#define	BUSCV_READ_FLAGS_WFFILE_EXIST	1
#define	BUSCV_READ_FLAGS_WFDATA		3

#define	BUSCV_DATA_TYPE_WFDISC		1
#define	BUSCV_DATA_TYPE_WFNOFILE	2
#define	BUSCV_DATA_TYPE_WFGAP		3
#define	BUSCV_DATA_TYPE_WFDATA		4

typedef struct buscvstachan_ {
	char sta[32];
	char chan[32];
	char stachan[64];
	char nscl[128];
	double tstart;
	double tend;
	Hook *dbmatches_hook;
	Tbl *wfdiscrows;
	Tbl *arrivalrows;
	Tbl *detectionrows;
} BUSCVStaChan;

typedef struct buscvtable_ {
	Dbptr db;
	long record_n;
	long sta_i;
	long sta_n;
	long chan_i;
	long chan_n;
	long time_i;
	long time_n;
	long endtime_i;
	long endtime_n;
	char *table_addr;
	Hook *dbmatches_hook;
	Tbl *matchlist1;
	Tbl *matchlist2;
	Tbl *rows;
} BUSCVTable;

typedef struct buscv_ {
	char *dbname;
	Dbptr db;
	Tbl *threads;
	Arr *tables;
	Tbl *stachan_accept_cache;
	Tbl *stachan_reject_cache;
	double tstart_cache;
	double tend_cache;
	Tbl *stachans_tbl;
	Arr *stachans_arr;
} BUSCV;

typedef struct buscvthreadstachan_ {
	Tbl *rows;
	double tstart;
	double tend;
} BUSCVThreadStaChan;

typedef struct buscvthread_ {
	int thread;
	BUSCV *scv;
	char table[32];
	Dbptr db;
	Tbl *rows;
	Arr *stachans_arr;
} BUSCVThread;

typedef struct buscvtrace_ {	/* Nominally corresponds to a single wfdisc row */
	Dbptr db;
	double tstart;
	double tend;
	double dt;
	long nsamp;
	double calib;			/* Converts input raw data to output
					   units. (e.g. nm/count). */
	double calper;
	char data_type[32];		/* "V", "A", etc. */
	float *data;			/* Points to data in input units before applying calib. */
	int data_size;			/* for reallocating data */
	Tbl *segments;
} BUSCVTrace;

typedef struct buscvtracesegment_ {
	int samp0;
	int nsamp;
	int data_type;
} BUSCVTraceSegment;

typedef struct bumoment_ {	/* seismic moment tensor object */
	double mxx, myy, mzz, mxy, mxz, myz;	/* original N-m moment tensor components in X=East, Y=North, Z=Up coordinate system */
	double isotropic;			/* isotropic moment tensor component */
	double mxx_d, myy_d, mzz_d;		/* deviatoric moment tensor trace */
	double eig1, eig2, eig3;		/* eigenvalues ordered lowest to highest */
	double eigv1[3], eigv2[3], eigv3[3];	/* corresponding eigenvectors ordered lowest to highest */
	double mxx_dc, myy_dc, mzz_dc, mxy_dc, mxz_dc, myz_dc;	/* double-couple N-m moment tensor components in X=East, Y=North, Z=Up coordinate system */
	double eig1_dc, eig2_dc, eig3_dc;	/* double-couple eigenvalues ordered lowest to highest */
	double eigv1_dc[3], eigv2_dc[3], eigv3_dc[3];	/* corresponding double-couple eigenvectors ordered lowest to highest */
	double p_axis[3], t_axis[3], n_axis[3]; /* P, T and N unit axes */
	double percent_dc;			/* percentage double couple */
	char *tag;
} BUMoment;

typedef struct bumoment_points_ {
	int npts;
	int npts_size;
	int outside_fill;
	double *xpts;
	double *ypts;
} BUMoment_points;

extern int splitlist (char *list, int *argc, char ***argv);
extern int splitlistc (char *list, int *argc, char ***argv, char *schars);
extern void freeargs (int argc, char **argv);

extern int parse_param (Pf *pf, char *name, int type, int need, void *val);

extern Pmtfifo *pmtfifo_create(int maxqueue, int block, int check);
extern int pmtfifo_set_timeout (Pmtfifo *mtf, double timeout);
extern int pmtfifo_flush (Pmtfifo *mtf, void (*free_value) ());
extern int pmtfifo_destroy (Pmtfifo *mtf, void (*free_value) ());
extern int pmtfifo_push (Pmtfifo *mtf, void *ptr);
extern int pmtfifo_unpop (Pmtfifo *mtf, void *ptr);
extern int pmtfifo_pop (Pmtfifo *mtf, void **ptr);
extern int pmtfifo_data_available (Pmtfifo *mtf);
extern int pmtfifo_release_block(Pmtfifo *mtf);
extern int pmtfifo_release_blocks (Pmtfifo *mtf);

extern Pmtmanagedfifo *pmtmanagedfifo_get (char *name);
extern Pmtmanagedfifo *pmtmanagedfifo_create(char *name);
extern Pmtfifo *pmtmanagedfifo_get_consumer (Pmtmanagedfifo *mtf, char *consumer_name);
extern Pmtfifo *pmtmanagedfifo_set_consumer (Pmtmanagedfifo *mtf, char *consumer_name, int maxqueue, int block);
extern void pmtmanagedfifo_free_data (void *data);
extern int pmtmanagedfifo_destroy (Pmtmanagedfifo *mtf);
extern int pmtmanagedfifo_push_eof (Pmtmanagedfifo *mtf, char *producer_name);
extern int pmtmanagedfifo_push (Pmtmanagedfifo *mtf, Hook *data, Hook * (*copyproc) (Hook *), int copydata);
extern int pmtmanagedfifo_try_push (Pmtmanagedfifo *mtf, Hook *data, Hook * (*copyproc) (Hook *), int copydata);
extern int pmtmanagedfifo_pop (Pmtmanagedfifo *mtf, char *consumer_name, Hook **data, unsigned long *sequence);
extern int pmtmanagedfifo_display (char *pre, Pmtmanagedfifo *mtf);
extern int pmtmanagedfifo_set_producer (Pmtmanagedfifo *mtf, char *producer_name);

extern Pmtcomp *pmtcomp_create(void);
extern long pmtcomp_destroy (Pmtcomp *mtc);
extern long pmtcomp_nthreads (Pmtcomp *mtc);
extern long pmtcomp_addthread (Pmtcomp *mtc, long (*comp_sub) (long ioffset, long number, void *user_data),
			long ioffset, long number, void *user_data);
extern long pmtcomp_setthread (Pmtcomp *mtc, long index, long (*comp_sub) (long ioffset, long number, void *user_data),
			long ioffset, long number, void *user_data);
extern long pmtcomp_deletethread (Pmtcomp *mtc, long index);
extern long pmtcomp_processthreads (Pmtcomp *mtc);

extern int pf2struc (Pf *pf, int ndefs, Pf2strucDef *defs, void *struc);

extern int buvector_get_heap_size(char *tag, char *handle, long *nobjects, long *heap_total);
extern BUVector *buvector_create(int nx, int ny, char *tag, char *handle);
extern BUVector *buvector_create_tsamp (int nsamp, double time, double samprate, 
					double *data, char *tag, char *handle);
extern BUVector *buvector_create_response_fap (char *filename, char *tag, char *handle, int *err);
extern BUVector *buvector_create_from_pf (Pf *pf, char *name, char *tag, char *handle, int *err);
extern BUVector *buvector_copy (char *in_handle, char *tag, char *handle);
extern BUVector *buvector_copy_1y (char *in_handle, int yindex, char *tag, char *handle);
extern BUVector *buvector_read (char *fname, char *tag, int line_off, int xpos, int ny, int *ypos, int lpos, char *handle);
extern BUVector *buvector_stats(Tbl *buvector_handles, char *type, char *tag, char *handle);
extern int buvector_read_from_trblob (char *trblob, char *tag, Dbptr *dbtr);
extern void buvector_free_vector (BUVector *vector);
extern int buvector_free (char *handle);
extern int buvector_free_tag (char *tag);
extern int buvector_resize (char *handle);
extern int buvector_append (char *handle, int index, double x, double *y, char *label);
extern int buvector_put (char *handle, int index, double x, double *y, char *label);
extern int buvector_puty (char *handle, int index, double *y, char *label);
extern int buvector_eval (char *handle, double x, double *y);
extern BUVector *buvector_get_vector (char *handle);
extern int buvector_get_point (char *handle, int index, double *x, double *y, char *label);
extern int buvector_get_points (char *handle, int *npts, int *ny, double **xpts, void *vypts, char ***labels );
extern int buvector_maxmin(char *handle, int iy, double *xmin, double *xmax, double *ymin, double *ymax);
extern int buvector_transform_edp (char *handle, double lonr, double latr);
extern int buvector_transform_invertx (char *handle);
extern int buvector_transform_spec (char *handle, int ndif, int iy);
extern int buvector_transform_ygain (char *handle, double ygain, int iy);

extern int buvector2_dump(char *handle);
extern int buvector2_get_heap_size (char *tag, char *handle, long *nobjects, long *heap_total, long *cache_objects, long *cache_total, long *nmallocs, long *nreallocs, long *nstrdups, long *nfrees);
extern int buvector2_set_cache (long max_size);
extern BUVector2 *buvector2_create(int nx, int ny, char *tag, char *handle);
extern BUVector2 *buvector2_create_tsamp (int nsamp, double time, double samprate, 
					double *data, char *tag, char *handle);
extern BUVector2 *buvector2_create_tsamp_from_db (Dbptr db, double t0, double t1, char *sta, char *chan, int apply_calib, int calib_from_wfdisc, double *calib, double *calper, char *segtype, char *tag, char *handle);
extern BUVector2 *buvector2_create_response_fap (char *filename, char *tag, char *handle, int *err);
extern BUVector2 *buvector2_create_from_pf (Pf *pf, char *name, char *tag, char *handle, int *err);
extern BUVector2 *buvector2_copy (char *in_handle, char *tag, char *handle);
extern BUVector2 *buvector2_copy_1y (char *in_handle, int yindex, char *tag, char *handle);
extern BUVector2 *buvector2_read (char *fname, char *tag, int line_off, int xpos, int ny, int *ypos, int lpos, char *handle);
extern BUVector2 *buvector2_stats(Tbl *buvector2_handles, char *type, char *tag, char *handle);
extern int buvector2_fill (char *handle, int nx, int ny, double *xpts, void *vypts, char **labels );
extern int buvector2_read_from_trblob (char *trblob, char *tag, Dbptr *dbtr);
extern void buvector2_free_vector (BUVector2 *vector);
extern int buvector2_free (char *handle);
extern int buvector2_free_tag (char *tag);
extern int buvector2_resize (char *handle);
extern int buvector2_append (char *handle, int index, double x, double *y, char *label);
extern int buvector2_put (char *handle, int index, double x, double *y, char *label);
extern int buvector2_puty (char *handle, int index, double *y, char *label);
extern int buvector2_eval (char *handle, double x, double *y);
extern BUVector2 *buvector2_get_vector (char *handle);
extern int buvector2_get_tsamp (char *handle, int *nsamp, double *tstart, double *dt, double **data, int *data_size );
extern int buvector2_get_point (char *handle, int index, double *x, double *y, char *label);
extern int buvector2_get_points (char *handle, int iy, int *nx, int *ny, double **xpts, int *xpts_size, double **ypts, int *ypts_size, char ***labels );
extern int buvector2_maxmin(char *handle, int iy, double *xmin, double *xmax, double *ymin, double *ymax);
extern int buvector2_maxmin2(char *handle, int iy, double xstart, double xend, double *xmin, double *xmax, double *ymin, double *ymax);
extern int buvector2_transform_edp (char *handle, double lonr, double latr);
extern int buvector2_transform_invertx (char *handle);
extern int buvector2_transform_spec (char *handle, int ndif, int iy);
extern int buvector2_transform_ygain (char *handle, double ygain, int iy);
extern int buvector2_filter (char *handle, char *filter_string);

extern void latlon2edp (double latr, double lonr, double lat, double lon, double *delx, double *dely);
extern void edp2latlon (double latr, double lonr, double delx, double dely, double *lat, double *lon);
extern void hls2rgb(double gamma,double h,double l,double s,double *r,double *g,double *b);

extern BUHistory *buhistory_get_history (char *handle);
extern BUHistory *buhistory_create(int maxnp, char *tag, char *filename, char *handle);
extern void buhistory_free_history (BUHistory *buhistory);
extern int buhistory_free (char *handle);
extern int buhistory_free_tag (char *tag);
extern int buhistory_add (char *handle, double time, double value);
extern int buhistory_eval (char *handle, double time, double *value);
extern int buhistory_maxmin (char *handle, double *tmin, double *tmax, double *vmin, double *vmax);
extern int buhistory_get_points (char *handle, int *npts, double **tpts, double **ypts);

extern BUDsseis *budsseis_get (char *handle);
extern int budsseis_create(Dbptr dbin, int argc, char **argv, BUDsseis **dsseis, char *handle);
extern void budsseis_free ( char *handle );
extern int budsseis_getnstachans (int argc, char **argv);
extern int budsseis_getstachan (int argc, char **argv, char *out);
extern int budsseis_getchannel (int argc, char **argv, int *nsegs, Dbptr *trace);
extern int budsseis_getsegment (Dbptr trace, int iseg, double *tstart, double *dt, int *nsamp, float **data);
extern int dsseis_getchannel (Dbptr dbsc, char *net, char *sta, char *chan, double tstart, double tend, 
	    char *gap, int calib, char *rotate, double strike, double dip, char *filter,
	    int *nsegs, Dbptr *trace);
extern int dsseis_grdb_sc_loadcss (Dbptr dbin, char *net_expr, char *sta_expr, char *chan_expr, double tstart, double tend,
		 int coords, int ir, int orient, Dbptr *dbscgr, Dbptr *dbsc);


extern BUVector * db2vector ( Dbptr db, char *xexpr, char *yexpr, char *hue_expr, char *sat_expr, char *siz_expr, char *sym_expr, double hue, double huerange, int fill, int max_size, char *tag, char *handle );
extern long buvector_nearest(char *handle, int iy, double x, double y) ;

extern BUGrid *bugrid_get_grid (char *handle);
extern long bugrid_register_grid (BUGrid *grid, char *handle);
extern BUGrid *bugrid_create(char *grid_type, long nxgrid, float *xgrid_coords, char *xgrid_type, char *xgrid_units, 
		long nygrid, float *ygrid_coords, char *ygrid_type, char *ygrid_units, 
		char *data_type, char *data_units, char *tag, char *handle);
extern void bugrid_free_bugrid (BUGrid *grid);
extern long bugrid_free (char *handle);
extern long bugrid_free_tag (char *tag);
extern BUGrid *bugrid_copy (char *in_handle, char *tag, char *handle);
extern long bugrid_put(char *handle, double time, long nsamps, double dt, float *data);
extern long bugrid_put_bugrid(BUGrid *grid, double time, long nsamps, double dt, float *data);
extern long bugrid_append_sample(char *handle, float *data);
extern long bugrid_append_sample_bugrid(BUGrid *grid, float *data);
extern long bugrid_trim(char *handle, int nsamp);
extern long bugrid_trim_bugrid(BUGrid *grid, int nsamp);
extern long bugrid_set_order(char *handle, int order);
extern long bugrid_set_order_bugrid(BUGrid *grid, int order);
extern long bugrid_putanddecimate(char *handle, double time, long nsamps, double dt, float *data, long decimation);
extern long bugrid_putanddecimate_bugrid(BUGrid *grid, double time, long nsamps, double dt, float *data, long decimation);
extern long bugrid_eval (char *handle, long isamp, long nxys, float *x, float *y, char *interp_mode,
			char *extrap_mode, float **out, long *out_size);
extern long bugrid_eval_bugrid (BUGrid *grid, long isamp, long nxys, float *x, float *y, char *interp_mode,
			char *extrap_mode, float **out, long *out_size);
extern long bugrid_order (BUGrid *grid);
extern long bugrid_maxmin(char *handle, long isamp, 
		float *xmin, float *xmax, float *ymin, float *ymax, float *zmin, float *zmax);
extern long bugrid_maxmin_bugrid(BUGrid *grid, long isamp, 
		float *xmin, float *xmax, float *ymin, float *ymax, float *zmin, float *zmax);
extern long bugrid_write_fd_bugrid(BUGrid *grid, int fd);
extern long bugrid_read_fd(int fd, BUGrid **grid);
extern long bugrid_eval_d2slow (char *handle, long isamp, long nds, float *distances, float *azimuths, 
		char *method, char *model, char *phase_code, float sdepth, char *interp_mode, char *extrap_mode, float **out, long *out_size);
extern long bugrid_eval_d2slow_bugrid (BUGrid *grid, long isamp, long nds, float *distances, float *azimuths, 
		char *method, char *model, char *phase_code, float sdepth, char *interp_mode, char *extrap_mode, float **out, long *out_size);
extern long bugrid_coords_d2slow (long nds, float *xcoords, float *ycoords, 
		char *method, char *model, char *phase_code, float sdepth, float gap_value);
extern long bugrid_stuff (char *handle, char **packet, int *nbytes, int *packetsz);
extern long bugrid_stuff_bugrid (BUGrid *grid, char **packet, int *nbytes, int *packetsz);
extern long bugrid_unstuff (char *packet, int nbytes, BUGrid **grid);

extern BUSac *busac_get_busac(char *handle);
extern BUSac *busac_create(char *tag, char *handle);
extern void busac_free_busac (BUSac *busac);
extern long busac_free(char *handle);
extern long busac_free_tag(char *tag);
extern long busac_puthdr_busac(BUSac *busac, ...);
extern long busac_gethdr_busac(BUSac *busac, ...);
extern long busac_putdata_busac(BUSac *busac, int nsamp, double tstart, double samprate, float *data, char *segtype);
extern long busac_getreftime_busac (BUSac *busac, double *rtime);
extern long busac_getdata_busac(BUSac *busac, int *nsamp, double *tstart, double *samprate, float **data, char *segtype);
extern long busac_putevent_busac(BUSac *busac, Dbptr dbtrace, Dbptr dborigin);
extern long busac_getevent_busac(BUSac *busac, Dbptr *dbtrace, Dbptr *dborigin);
extern long busac_write_to_file_busac(BUSac *busac, char *filename);
extern BUSac *busac_read_from_file_busac(char *filename, char *tag, char *handle);
extern void busac_free_header_tbl (void *element);
extern int busac_header_2_tbl (BUSac *busac, Tbl **tbl);
extern int busac_tbl_2_header (Tbl *tbl, BUSac *busac);

extern int buscv_create(int argc, char **argv, char *handle);
extern int buscv_destroy (char *handle);
extern void buscv_free_index (Dbptr db, Tbl *pattern);
extern int buscv_subset_cache (char *handle, Tbl *stachan_accept, Tbl *stachan_reject, double tstart, double tend);
extern int buscv_subset (char *handle, char *stachan, double tstart, double tend, int getwf, int getarr, int getdet, BUSCVStaChan **sc);
extern int buscv_subset_refresh_wfdisc (char *handle);
extern int buscv_subset_refresh_arrival (char *handle);
extern int buscv_subset_refresh_detection (char *handle);
extern int buscv_subset_refresh (char *handle);
extern int buscv_create_cache (char *handle);
extern BUSCV *buscv_get (char *handle);
extern int buscv_get_latlonelev (BUSCV *scv, BUSCVStaChan *sc, double time, double *lat, double *lon, double *elev);
extern int buscv_get_sampratecalib (BUSCV *scv, BUSCVStaChan *sc, Dbptr dbwfdisc, int calib_from_db, double *samprate, double *calib, double *calper, char *segtype);
extern int buscv_get_response (BUSCV *scv, BUSCVStaChan *sc, double time, Response **response, double *calib, double *calper, char *segtype);
extern void buscv_get_nscl (BUSCV *scv, char *sta, char *chan, char *nscl);
extern void buscv_trace_free_table (Tbl *trace_tbl);
extern void buscv_trace_free (BUSCVTrace *trace);
extern int buscv_trace_get_table (BUSCV *scv, BUSCVStaChan *sc, double tstart, double tend, int nosamples, int calib_from_db, Tbl **trace_tbl);
extern int buscv_trace_process_gaps (Tbl *trace_tbl, char *how);

extern BUMoment *bumoment_create (double mxx, double xyy, double mzz, double mxy, double mxz, double myz, char *tag, char *handle);
extern BUMoment *bumoment_get_moment (char *handle);
extern void bumoment_free_bumoment (BUMoment *moment);
extern int bumoment_compute_bb_points (char *handle, BUMoment_points *points_P1_outline, BUMoment_points *points_P2_outline, 
							BUMoment_points *points_P1_fill, BUMoment_points *points_P2_fill, 
							BUMoment_points *points_DC1_outline, BUMoment_points *points_DC2_outline,
							int dont_make_outside_full);
extern int bumoment_compute_bb (char *handle, char *vecP1_handle_outline, char *vecP2_handle_outline, 
			char *vecP1_handle_fill, char *vecP2_handle_fill, char *vecDC1_handle_outline, char *vecDC2_handle_outline);
extern void bumoment_free_bumoment (BUMoment *moment);
extern BUMoment_points *bumoment_points_create ();
extern void bumoment_points_init (BUMoment_points *points);
extern void bumoment_points_clear (BUMoment_points *points);
extern void bumoment_points_free (BUMoment_points *points);
extern void bumoment_points_reset (BUMoment_points *points);

#ifdef  __cplusplus
}
#endif

#endif
