//   Copyright (c) 2019 Boulder Real Time Technologies, Inc.           
//                                                                     
//   This software module is wholly owned by Boulder Real Time         
//   Technologies, Inc. Any use of this software module without        
//   express written permission from Boulder Real Time Technologies,   
//   Inc. is prohibited.                                               

#ifndef OORB_H
#define OORB_H

#include <iostream>
#include <sstream>
#include <thread>
#include <mutex>
#include <vector>
#include <regex>
#include <condition_variable>
#include <unordered_map>
#include <list>
#include <algorithm>

#include "signal.h"

#include "stock.h"
#include "orb.h"
#include "brttutil.h"
#include "brttpkt.h"
#include "sysdata.h"
#include "BUConfigure.h"

#include "oorb_taverage.h"
#include "oorb_sourcecounter.h"

#define EPOCHTIME_NULL          -9999999999.999
#define EPOCHTIME_LAST          -8888888888.888 
#define EPOCHTIME_STRING_FORMAT (char*) "%D %T %Z" 

#define block_signals()	{ sigset_t set; \
			  sigemptyset( &set ); \
			  sigaddset( &set, SIGHUP ); \
			  sigaddset( &set, SIGINT ); \
			  sigaddset( &set, SIGQUIT ); \
			  sigaddset( &set, SIGTERM ); \
			  sigaddset( &set, SIGUSR1 ); \
			  pthread_sigmask( SIG_BLOCK, &set, NULL ); }

#define await_signals(S) { sigset_t set; \
			  sigemptyset( &set ); \
			  sigaddset( &set, SIGHUP ); \
			  sigaddset( &set, SIGINT ); \
			  sigaddset( &set, SIGQUIT ); \
			  sigaddset( &set, SIGTERM ); \
			  sigaddset( &set, SIGUSR1 ); \
			  pthread_sigmask( SIG_SETMASK, &set, NULL ); \
			  sigwait( &set, S ); }

namespace Oorb {
	class 	Exception : public std::runtime_error { 
			public:
										Exception( std::string what ) : std::runtime_error( what ) {};
										~Exception( void ) {};
		};	
	class 	EOFException : public Exception { 
			public:
										EOFException( std::string what ) : Exception( what ) {};
										~EOFException( void ) {};
		};	
	class 	StuffException : public Exception { 
			public:
										StuffException( std::string what ) : Exception( what ) {};
										~StuffException( void ) {};
		};	
	struct 	LogMsg {
										LogMsg( void );
										~LogMsg( void );

				std::string					msg;
				std::string					caller;
				std::thread::id 				tid;
				bool 						includeSyserr;
				enum Severity : int { 
				       QUIET = -1,
				       NOTIFY = 0, 
				       VERBOSE,
				       VERBOSE_VERY,
				       VERBOSE_EXTREMELY,
				       VERBOSE_DUMP,
				       VERBOSE_TORRENTIALLY,
				       COMPLAIN,
				       DIE,
				       DIE_INITIALIZING,
				       DEBUG,
				       SCAFFOLD,
				       SCAFFOLD_FEATURE,
				       UNSET
				       } 					severity;
				Severity 					verbosity_task;

				static Severity					incrementVerbose( Severity severity );
				static Severity 				string2severity( std::string severity_string );
				static std::string 				severity2string( Severity severity );
				static std::string 				severity2abbreviation( Severity severity );
		};
	class Task;	// Forward declaration for Command
	struct 	Command {
										Command( void );
										~Command( void );
				enum { NULLCOMMAND,
				       LOG,
			       	       STARTCONNECTION,
			       	       STARTTASK,
			       	       STOPTASK,
			       	       USERCOMMAND
			       	       }					command;
				LogMsg* 					logMsg;
				Pf* 						pf;
				std::shared_ptr<Task> 				task;
		};
	static int 
	boolParserProc( BU_ConfigureSpec* spec, char* value, char* paramRec, void* object_ptr ) {
		bool 	bval_in;
		bool 	bval;

		if( strcmp( spec->type, "bool" ) ) {

			return BU_CONFIGURE_UNRECOGNIZED_TYPE;

		} else if( ! ( spec->specFlags & BU_CONFIGURE_NULL_OK ) && value == NULL ) {

			bu_register_error( 0, "Oorb::boolParserProc: Cannot set NULL value if not BU_CONFIGURE_NULL_OK for '%s'.\n", spec->name );

			return BU_CONFIGURE_ERROR;
		}

		int ival;

		if( BU_ParseBoolean( value, &ival ) < 0 ) {

			bu_register_error( 0, "Oorb::boolParserProc: BU_ParseBoolean() error for '%s'.\n", spec->name );

			return BU_CONFIGURE_ERROR;
		}

		bval = (bool) ival;

		bval_in = *(bool*)( paramRec + spec->offset );

		*(bool*)( paramRec + spec->offset ) = bval;

		if( bval_in != bval ) spec->specFlags |= BU_CONFIGURE_OPTION_SET;

		return BU_CONFIGURE_OK;
	}
	static int 
	timeParserProc( BU_ConfigureSpec* spec, char* value, char* paramRec, void* object_ptr ) {
		double 	e_in;
		double 	e;

		if( strcmp( spec->type, "time" ) ) {

			return BU_CONFIGURE_UNRECOGNIZED_TYPE;

		} else if( value == NULL || ! strcmp( value, "" ) || ! strcmp( value, "NULL" ) ) {

			if( ! ( spec->specFlags & BU_CONFIGURE_NULL_OK ) ) {

				bu_register_error( 0, "Oorb::timeParserProc: Cannot set NULL value if not BU_CONFIGURE_NULL_OK for '%s'.\n", spec->name );

				return BU_CONFIGURE_ERROR;

			} else {

				e = EPOCHTIME_NULL;
			}

		} else if( ! strcmp( value, "last" ) ) {

			e = EPOCHTIME_LAST;

		} else if( ( ! strcmp( value, "0" ) ) || ( ! strcmp( value, "0.0" ) ) ) {

			e = 0.0;

		} else if( ! is_epoch_string( value, &e ) ) {

			bu_register_error( 0, "Oorb::timeParserProc: is_epoch_string() error for '%s'.\n", spec->name );

			return BU_CONFIGURE_ERROR;
		}

		e_in = *(double*)( paramRec + spec->offset );

		*(double*)( paramRec + spec->offset ) = e;

		if( e_in != e ) spec->specFlags |= BU_CONFIGURE_OPTION_SET;

		return BU_CONFIGURE_OK;
	}
	static int 
	timedeltaParserProc( BU_ConfigureSpec* spec, char* value, char* paramRec, void* object_ptr ) {
		double 	d_in;
		double 	d;
		int 	rc;

		if( strcmp( spec->type, "timedelta" ) ) {

			return BU_CONFIGURE_UNRECOGNIZED_TYPE;

		} else if( value == NULL || ! strcmp( value, "" ) || ! strcmp( value, "NULL" ) ) {

			if( ! ( spec->specFlags & BU_CONFIGURE_NULL_OK ) ) {

				bu_register_error( 0, "Oorb::timedeltaParserProc: Cannot set NULL value if not BU_CONFIGURE_NULL_OK for '%s'.\n", spec->name );

				return BU_CONFIGURE_ERROR;

			} else {

				d = EPOCHTIME_NULL;
			}

		} else if( ( rc = sscanf( value, "%lf", &d ) ) != 1 ) {

			bu_register_error( 0, "Oorb::timedeltaParserProc: sscanf() error (value= '%s', rc=%d) for '%s'.\n", value, rc, spec->name );

			return BU_CONFIGURE_ERROR;
		}

		d_in = *(double*)( paramRec + spec->offset );

		*(double*)( paramRec + spec->offset ) = d;

		if( d_in != d ) spec->specFlags |= BU_CONFIGURE_OPTION_SET;

		return BU_CONFIGURE_OK;
	}
	template <class T>
	class 	Configure : public BUConfigure {
			public:
										Configure( BU_ConfigureSpec* specs ) 
											: m_configureSpecs( specs )
					{
						memset( &(this->m_params), 0, sizeof( T ) );

						addTypeParser( (char*) "bool", boolParserProc );
						addTypeParser( (char*) "time", timeParserProc );
						addTypeParser( (char*) "timedelta", timedeltaParserProc );

						this->setConfigureSpecs( m_configureSpecs, (char*) &(this->m_params), NULL );
					
						return;
					}

										~Configure( void ) { return; }


				T 						m_params;

			private:

				BU_ConfigureSpec* 				m_configureSpecs;
		};
	class 	Orb { 
			public: 
										Orb( std::string orbname );
										~Orb( void );

				std::string 					m_orbname;	
				bool 						m_is_forbserver;
		};
	class 	OrbPacket { 
			public:
				friend class Queue;
				friend class CalibrationMgrTask;

										OrbPacket( std::string source, int pktid, char* srcname, double pkttime, char* pkt, int nbytes );
										OrbPacket( std::string srcname, Pf* pf );
										OrbPacket( std::string srcname, std::string astring, bool is_internal_log = true ); // (currently true)
										OrbPacket( const OrbPacket& op );
										~OrbPacket( void );

				std::string 					describe( void ) const;
				bool 						isWf( void );
				bool 						unstuffs( bool suppress_unstuff_errors = false );
				bool 						stuffs( void );
				double 						nsec_max( void );
				char*						getPktBufCopy( void );
				const Pf* 					getReadOnlyPf( void );
				std::string 					getSuffix( void );
				bool 			 			rename( std::string newSrcname );

				char 						m_srcname[ORBSRCNAME_SIZE];
				double 						m_pkttime;
				int 						m_pktid;
				int 						m_nbytes;
				bool 						m_is_internal_log;

			protected: 

				Packet* 					m_pkt;

			private: 

				int 						srcname2content( std::string srcname );
				void 						unstuff( bool suppress_unstuff_errors = false );

				typedef struct OrbPacketHookData_ { 
						std::shared_ptr<OrbPacket> op; 
						} 				OrbPacketHookData;

				static void					spOrbPacket_hookfree( void* p );
				static Hook* 					spOrbPacket_hookcopy( Hook* h );

				int 						m_bufsize;
				char* 						m_pktbuf;
				int 						m_content;
				std::string 					m_packetSource;
				bool 						m_unstuffFailed;
		};
	class 	StateVector {
			public:
										StateVector( void );
										~StateVector( void );

				void 						setTaskName( std::string taskName );
				void 						setOrbName( std::string orbname );
				void 						setOrbStart( double starttime );
				void 						recordLastPacketId( int pktid );
				void 						recordLastPacketTime( double pkttime );

				Pf* 						createPfSnapshot( void );

				std::string 					m_taskName;
				std::string 					m_orbname;
				int 						m_last_pktid;
				double 						m_last_pkttime;
				double 						m_max_pkttime;
				double 						m_orb_start; 				
				double 						m_last_update;
				bool 						m_buryStateInfo;

			private:

				std::mutex 					m_stateVectorMutex;
				int 						m_npkts_unsnapshotted;
				double 						m_pfsnapshottime;
				Pf* 						m_pfsnapshot;
		};
	class 	StatusVector {
			public:
										StatusVector( void );
										~StatusVector( void );

				void 						addNewTimeAverage( std::string varname, double time, double tbin, int nbin );
				void 						addNewSourceCounter( std::string varname, double twin );

				void 						getTimeAverage( std::string varname, double* time, double* twin, double* sum, double* ave );
				long 						getSourceCount( std::string varname );
				void 						putTimedValue( std::string varname, double value );
				void 						putSourceName( std::string varname, std::string srcname, double time );
				void 						putBasics( std::shared_ptr<OrbPacket> op );
				Pf* 						getTimeAveragePfHistory( std::string varname );
				Pf* 						getSourceCounterPfHistory( std::string varname );

				char* 						createStringPfSnapshot( std::string taskName );
				Pf* 						createStatusStatePfSnapshot( void );
				void						restoreStatus( Pf* pf );

			private: 

				std::recursive_mutex 				m_statusVectorMutex; 
				std::unordered_map<std::string,TAverage*>	m_timeAverages;		// Indexed by string name of tracked variable
				std::unordered_map<std::string,SourceCounter*>	m_sourceCounters;	// Indexed by string name of tracked variable
		};
	class   OrbConnection {
			public:
				typedef enum Direction { READ, WRITE } 		Direction;

										OrbConnection( std::shared_ptr<Orb> orb, Direction direction );
										~OrbConnection( void );

				bool 			 			open( void );
				bool 						isOpen( void );
				void 						close( void );

				int 						select( std::string match );
				int 						reject( std::string reject );
				int 						after( double atime );
				void 						smartReposition( std::string taskName, double starttime );

				std::shared_ptr<OrbPacket> 			reap_timeout( double maxseconds );
				std::shared_ptr<OrbPacket> 			get( int whichpkt);
				int 						put( std::shared_ptr<OrbPacket>, bool acknowledge = false );

				int 						getNewestPktid( void );

				std::shared_ptr<Orb>				m_orb;
				std::shared_ptr<StateVector>			m_stateVector;
				std::shared_ptr<StatusVector>			m_statusVector;
				int 						m_orbfd;
				Direction 					m_direction;
		};
	class 	Queue {
			public: 	
										Queue( std::string queueName );
										~Queue( void );

				void 						setProducer( std::string producerName );
				void 						setConsumer( std::string consumerName, int maxQueue );

				void						try_push( std::shared_ptr<OrbPacket>& op );
				void 						push( std::shared_ptr<OrbPacket>& op );
				void 						push_eof( std::string producerName );
				std::shared_ptr<OrbPacket> 			pop( std::string consumerName );
				int 						getQueueCount( std::string consumerName );

				std::string 					m_queueName;

			private:

				Pmtmanagedfifo* 				m_fifo;
				std::unordered_map<std::string,Pmtfifo*> 	m_consumers;										

		};
	class 	Filter {
			public:
										Filter( bool winnow, bool transform, std::shared_ptr<StatusVector> statusVector = nullptr );
										~Filter( void );

				std::shared_ptr<OrbPacket>			filter( std::shared_ptr<OrbPacket> op );

			protected:

				virtual bool					accept( std::shared_ptr<OrbPacket> op );
				virtual std::shared_ptr<OrbPacket>		morph( std::shared_ptr<OrbPacket> op );

				bool 						m_winnowPackets;
				bool 						m_transformPackets;
				std::shared_ptr<StatusVector>			m_statusVector;
		};
	class 	FilterCascade {
			public: 
										FilterCascade( void );
										~FilterCascade( void );

				void 						addFilter( Filter* f );

				std::shared_ptr<OrbPacket>			filter( std::shared_ptr<OrbPacket> op );

			private: 

				std::list<Filter*> 				m_filters;
		};
	class 	UnstuffFilter : public Filter {
			public:
										UnstuffFilter( bool check_unstuff, bool suppress_unstuff_errors, std::shared_ptr<StatusVector> );
										~UnstuffFilter( void );

			protected:

				bool						accept( std::shared_ptr<OrbPacket> op );

			private: 

				bool 						m_check_unstuff;
				bool 						m_suppress_unstuff_errors;
		};
	class 	EndtimeFilter : public Filter {
			public:
										EndtimeFilter( double endtime, int lastPktidOnOrbAtStart, bool* acquiredLastPacket );
										~EndtimeFilter( void );

			protected:

				bool						accept( std::shared_ptr<OrbPacket> op );

			private: 

				double 						m_endtime;
				int 						m_lastPktidOnOrbAtStart;
				bool* 						m_acquiredLastPacket;
		};
	class 	TooOldFilter : public Filter {
			public:
										TooOldFilter( double too_old, std::shared_ptr<StatusVector> );
										~TooOldFilter( void );

			protected:

				bool						accept( std::shared_ptr<OrbPacket> op );

			private: 

				double 						m_too_old;
		};
	class 	TooNewFilter : public Filter {
			public:
										TooNewFilter( double too_new, std::shared_ptr<StatusVector> );
										~TooNewFilter( void );

			protected:

				bool						accept( std::shared_ptr<OrbPacket> op );

			private: 

				double 						m_too_new;
		};
	class 	MatchRejectFilter : public Filter {
			public:
										MatchRejectFilter( std::string match, std::string reject, std::shared_ptr<StatusVector> );
										~MatchRejectFilter( void );

			protected:

				bool						accept( std::shared_ptr<OrbPacket> op );

			private: 

				std::string 					m_match;
				std::string 					m_reject;
				std::regex 					m_matchRegex;
				std::regex 					m_rejectRegex;
		};
	class 	TranslateNamesFilter : public Filter {
			public:
										TranslateNamesFilter( std::string translations, std::shared_ptr<StatusVector> );
										~TranslateNamesFilter( void );

			protected:

				std::shared_ptr<OrbPacket>			morph( std::shared_ptr<OrbPacket> op );

			private: 

				std::string					m_translations;
		};
	class 	RemapHyphensFilter : public Filter {
			public:
										RemapHyphensFilter( bool remap_hyphens, std::shared_ptr<StatusVector> );
										~RemapHyphensFilter( void );

			protected:

				std::shared_ptr<OrbPacket>			morph( std::shared_ptr<OrbPacket> op );

			private: 

				bool						m_remap_hyphens;
		};
	class 	AddCalibrationFilter : public Filter {
			public:
										AddCalibrationFilter( bool add_calibration, std::shared_ptr<StatusVector> );
										~AddCalibrationFilter( void );

			protected:

				std::shared_ptr<OrbPacket>			morph( std::shared_ptr<OrbPacket> op );

			private: 

				bool 						m_add_calibration;
		};
	typedef struct taskParamsStruct {
				LogMsg::Severity 				verbosity_task;
				bool 						run;
	} taskParamsStruct;
	class 	Task : public Configure<taskParamsStruct> { 
			public:
										Task( std::string taskName );
										~Task( void );

				virtual void					run( void ) = 0;
				void 						awaitTaskReady( void );
				void 						awaitTaskStopped( double limit_sec );
				void 						requestTaskStop( void );
				void 						pushCommand( Command* command );
				bool 						stopping( void );

				std::string					m_taskName;
				std::thread::id 				m_taskThreadId;

			protected:

				void 						init_run( void );
				bool 						runconfig( void );
				void 						announceTaskReady( void );
				void 						announceTaskStopped( void );
				void 						spinSleep( bool& spinning );
				void 						finalize_run( void );
				bool 						commandAvailable( void );
				Command* 					popCommand( void );
				int 						processCommands( void );
				virtual void 					processCommand( Command* command );

				std::mutex 					m_taskMutex;
				bool 						m_taskReady;
				std::mutex 					m_taskReadyMutex;
				std::condition_variable 			m_taskReadyCV;
				bool 						m_taskStopped;
				std::mutex 					m_taskStoppedMutex;
				std::condition_variable 			m_taskStoppedCV;
				Pmtfifo* 					m_commandFifo;	
				bool 						m_stopTask;
		};
	class 	ConfigMgrTask : public Task {
			public:
										ConfigMgrTask( int argc, char** argv, std::string programName = "" );
										~ConfigMgrTask( void );

				virtual void					run( void ) = 0;	
				virtual std::string 				usage( void ) = 0;

				int 						m_argc;
				char** 						m_argv;	
				std::string 					m_programName;
				std::string 					m_commandLine;

		};
	typedef struct logParamsStruct {
				char* 						write_to_queue;
	} logParamsStruct;
	class 	LogMgrTask : public Task {

			public:
										LogMgrTask( void );
										~LogMgrTask( void );

				void 						run( void );

				void 						notify( std::string msg, std::string caller = "", bool includeSyserr = false );
				void 						verbose( std::string msg, std::string caller = "", bool includeSyserr = false );
				void 						vVerbose( std::string msg, std::string caller = "", bool includeSyserr = false ); // Very
				void 						xVerbose( std::string msg, std::string caller = "", bool includeSyserr = false ); // Extremely
				void 						dVerbose( std::string msg, std::string caller = "", bool includeSyserr = false ); // Dump
				void 						tVerbose( std::string msg, std::string caller = "", bool includeSyserr = false ); // Torrentially

										// Caps to avoid macro clash with Antelope complain() and die():
				void 						Complain( std::string msg, std::string caller = "", bool includeSyserr = false ); 
				void 						Die( std::string msg, std::string caller = "", bool includeSyserr = false );

				void 						die_initializing( bool printUsage, std::string msg, std::string caller = "", bool includeSyserr = false );
				void 						DEBUG( std::string msg, std::string caller = "", bool includeSyserr = false );
				void 						SCAFFOLD( std::string msg, std::string caller = "", bool includeSyserr = false );
				void 						SCAFFOLD_FEATURE( std::string msg, std::string caller = "", bool includeSyserr = false );

				void 						startMsg( void );
				void 						finishMsg( void );
				void 						flush( void );

										// Necessary to cache task names and verbosities so they're available 
										// to LogMgrTask even as the task's thread is being decomissioned and joined:
				void 						cacheTaskDetails( Task* task );
				void 						cacheTaskDetails( std::shared_ptr<Task> task );

				std::string 					getTaskName( std::thread::id tid );	

				logParamsStruct 				m_logParams;

				static char* 					shifted_string( char* s );
				static char* 					pf2shifted_string( Pf* pf );

			protected: 

				void 						processCommand( Command* command );

			private:
			
				void 						pushLogMsg( LogMsg* logMsg );
				void 						processLogMsg( LogMsg* logMsg );

				LogMsg::Severity 				getTaskVerbosity( std::string taskName );

				int 						m_longestTaskNameLength;
				int 						m_longestCallerNameLength;

				std::shared_ptr<Queue>				m_queue;

				std::unordered_map<std::thread::id,std::string>	m_taskNames;	
				std::unordered_map<std::string,LogMsg::Severity> m_taskVerbosities;	
		};
	class 	SignalMgrTask : public Task {
			public:
										SignalMgrTask( void );
										~SignalMgrTask( void );

				void 						run( void );
		};
	typedef struct ExtraStateBlock {
				std::string 					key;
				void  						(*read_block)( Pf* block, void* pvt );
				Pf* 						(*write_block)( void* pvt );
				void* 						pvt;
	} ExtraStateBlock;
	typedef struct stateParamsStruct {
				char* 						statefile;
				double 						retain_unused_stateinfo_sec;
	} stateParamsStruct;
	class 	StateMgrTask : public Task {
			public:
										StateMgrTask( void );
										~StateMgrTask( void );

				void 						run( void );
				std::shared_ptr<StateVector>			getStateVector( std::string taskName, std::string orbname );
				void 						registerExtraStateBlock( std::string key, 
													 void (*read_block)(Pf*,void*),
													 Pf* (*write_block)(void*),
													 void* pvt );

				stateParamsStruct 				m_stateParams;	

			private:
			
				void 						read_state( void );
				void 						write_state( void );

				void 						protestStaleState( void );	
				std::shared_ptr<StateVector> 			pf2StateVector( std::string key, Pf* pf, double statefileTime );
				ExtraStateBlock*				isExtraBlock( std::string key );

				std::unordered_map<std::string,std::shared_ptr<StateVector>> m_exhumedStateVectors;	// Indexed by state-pf key
				std::unordered_map<std::string,std::shared_ptr<StateVector>> m_activeStateVectors;	// Indexed by taskName
				std::unordered_map<std::string,ExtraStateBlock*> m_extraStateBlocks;			// Indexed by statefile key for extra block
		};
	typedef struct statusParamsStruct {
				char* 						write_to_queue;
				double 						pfstatusreport_sec;
				double 						datarate_interval_sec;
				double 						thruput_interval_sec;
	} statusParamsStruct;
	class 	StatusMgrTask : public Task {
			public:
										StatusMgrTask( void );
										~StatusMgrTask( void );

				void 						run( void );
				std::shared_ptr<StatusVector>			getStatusVector( std::string taskName );
				void 						addStatusVector( std::string taskName, std::shared_ptr<StatusVector> statusVector );
				void 						attemptForceWrite( void );

				statusParamsStruct 				m_statusParams;
				std::shared_ptr<Queue>				m_queue;

			private: 

				bool 						initializeQueue( void );
				std::shared_ptr<OrbPacket> 			pfstatus( void );
				void 						write_status( void );

				double 						m_lastReport;

				std::unordered_map<std::string,std::shared_ptr<StatusVector>> m_statusVectors;	// Indexed by taskName
		};
	typedef struct commandParamsStruct {
				char* 						orbtag;
				char* 						orbname;
	} commandParamsStruct;
	class 	CommandMgrTask : public Task {
			public:
				typedef enum State { CONNECTING_READ, 
						     CONNECTING_WRITE,
						     CONFIGURING,
						     RUNNING }			State;

										CommandMgrTask( void );
										~CommandMgrTask( void );

				void 						run( void );
				Pf* 						executeAndRespond( const Pf* pf_command );

				commandParamsStruct 				m_commandParams;
				std::shared_ptr<OrbConnection> 			m_readerConnection;
				std::shared_ptr<OrbConnection> 			m_writerConnection;
				State 						m_taskState;

			private: 

				int 						m_sequence_target;
		};
	typedef struct calibrationParamsStruct {
				char* 						dbname;
				bool 						onlynull;
				int 						recheck_npackets;
	} calibrationParamsStruct;
	class 	CalibrationMgrTask : public Task {
			public:
										CalibrationMgrTask( void );
										~CalibrationMgrTask( void );

				void 						run( void );

				std::shared_ptr<OrbPacket> 			addCalib( std::shared_ptr<OrbPacket> op );

				calibrationParamsStruct 			m_calibrationParams;

			private: 

				PktChannelCalib* 				m_pcc;
		};
	typedef struct connectionParamsStruct {
				char* 						match_task;
				char* 						reject_task;
				char*						direction; 	// redundant, supports parsing
				char* 						name;		// redundant, supports parsing
	} connectionParamsStruct;
	class 	OrbConnectionTask : public Task {
			public:
				typedef enum State { CONNECTING, 
						     CONFIGURING,
						     COMMUNICATING,
						     PAUSED }			State;

										OrbConnectionTask( std::string taskName, OrbConnection::Direction direction );
										~OrbConnectionTask( void );

				void 						requestTaskPause( void );
				void 						requestTaskContinue( void );

				static bool					vetConnectionPf( Pf* pf );

				connectionParamsStruct 				m_connectionParams;
				OrbConnection::Direction			m_direction;
				std::shared_ptr<OrbConnection> 			m_orbConnection;
				std::shared_ptr<Queue>				m_queue;
				State 						m_taskState;
				State 						m_prepauseState;
				FilterCascade 					m_filterCascade;
		};
	typedef struct readerParamsStruct {
				char* 						read_from_orbname;
				char* 						read_from_orbtag;
				char* 						write_to_queue;
				char* 						translate_names;
				bool 						check_unstuff;
				bool 						suppress_unstuff_errors;
				bool 						add_calibration;
				bool						remap_hyphens;
				double 						acq_starttime_task;
				double 						acq_endtime_task;
				double 						too_old;
				double 						too_new;

	} readerParamsStruct;
	class 	OrbReaderTask : public OrbConnectionTask {
			public:
										OrbReaderTask( std::string taskName, Pf* pf );
										~OrbReaderTask( void );

				void 						run( void );

				readerParamsStruct 				m_readerParams;
				int 						m_lastPktidOnOrbAtStart;

				static int 					s_numActiveReaders;

		};
	typedef struct writerParamsStruct {
				char* 						read_from_queue;
				char* 						write_to_orbname;
				char* 						write_to_orbtag;
				char* 						clone;
				int 						max_queue;
				bool 						acknowledge;
	} writerParamsStruct;
	class 	OrbWriterTask : public OrbConnectionTask {
			public:
										OrbWriterTask( std::string taskName, Pf* pf );
										~OrbWriterTask( void );

				void 						run( void );

				writerParamsStruct 				m_writerParams;
		};
	typedef struct masterParamsStruct {
				double 						internal_timeout_sec;
				double 						shutdown_grace_period_sec;
				double 						acq_starttime_program;
				double 						acq_endtime_program;
				char*						match_program;
				char*						reject_program;
				char* 						target_name;
				LogMsg::Severity 				verbosity_program;
				bool 						acknowledge_program;
	} masterParamsStruct;
	class 	Master : public Task {
			public:
										Master( std::shared_ptr<ConfigMgrTask> config );
										~Master( void );

				void 						run( void );

				void 						launchTask( std::shared_ptr<Oorb::Task> task );
				void 						incrementVerbose( void );
				void 						registerTask( Task* task );
				void 						deregisterTask( std::string taskName );
				void 						requestProgramStop( void );

				void 						addOrb( std::string orbname );
				void 						tagOrb( std::string orbname, std::string orbtag );
				void 						addTaggedOrb( std::string orbname, std::string orbtag );
				void 						addConnection( Pf* pf );
				void 						addTask( std::shared_ptr<Task> task );
				bool 						addClones( Pf* pf, char* tblname );
				bool 						addTranslations( Pf* pf, char* tblname );

				std::shared_ptr<Orb> 				getOrbByName( std::string orbname );
				std::shared_ptr<Orb> 				getOrbByTag( std::string orbtag );
				std::shared_ptr<Orb> 				getOrb( std::string orbtag, std::string orbname );
				std::shared_ptr<Queue>				getQueue( std::string queueName );
				std::shared_ptr<Task>				getTask( std::string taskName );
				std::shared_ptr<Task>				getTask( std::thread::id tid );
				std::shared_ptr<Task>				getMyTask( void );
				std::shared_ptr<OrbConnectionTask>		getConnectionTask( std::string taskName );
				Table<std::string> 				getClones( char* tblname, char* srcname );
				std::string 					getTranslation( char* tblname, char* srcname );

				std::shared_ptr<CalibrationMgrTask>		calibration( void );
				std::shared_ptr<CommandMgrTask>			commands( void );
				std::shared_ptr<ConfigMgrTask>			config( void );
				std::shared_ptr<LogMgrTask>			log( void );
				std::shared_ptr<SignalMgrTask>			signal( void );
				std::shared_ptr<StateMgrTask>			state( void );
				std::shared_ptr<StatusMgrTask>			status( void );

				masterParamsStruct 				m_masterParams;	
				double 						m_program_starttime;
				pid_t 						m_pid;

				std::unordered_map<std::string,std::shared_ptr<OrbConnectionTask>>	m_connectionTasks; 
				std::unordered_map<std::string,std::shared_ptr<Task>>			m_customTasks; 

			protected:

				void 						processCommand( Command* command );

			private:
				void 						addThread( std::thread t );
				void 						joinThreads( void );
				void 						startConnection( Pf* pf );
				void 						startTask( std::shared_ptr<Task> task );

				static bool					s_instantiated;

				std::shared_ptr<CalibrationMgrTask>		m_task_calibration;
				std::shared_ptr<CommandMgrTask>			m_task_command;
				std::shared_ptr<ConfigMgrTask>			m_task_config;
				std::shared_ptr<LogMgrTask> 			m_task_log;
				std::shared_ptr<SignalMgrTask> 			m_task_signal;
				std::shared_ptr<StateMgrTask>			m_task_state;
				std::shared_ptr<StatusMgrTask> 			m_task_status;

				std::unordered_map<std::thread::id,std::thread>		m_threads;	
				std::unordered_map<std::string,std::shared_ptr<Task>>	m_tasks;
				std::unordered_map<std::string,std::shared_ptr<Orb>>	m_orbs;
				std::unordered_map<std::string,std::string>		m_orbtags;
				std::unordered_map<std::string,std::shared_ptr<Queue>>	m_queues;
				std::unordered_map<std::string,std::list<Morphtbl*>>	m_clonesets;
				std::unordered_map<std::string,Morphtbl*>		m_xlatsets;
		};
}

extern std::shared_ptr<Oorb::Master> Om;

#endif		 
