/* Copyright (c) 2014 Boulder Real Time Technologies, Inc. */
/* All rights reserved */

#ifndef __bueventview__
#define __bueventview__

/* This software module is wholly owned by Boulder Real Time 
   Technologies, Inc. Any use of this software module without
   express written permission from Boulder Real Time Technologies,
   Inc. is prohibited. */

#include "BUConfigure.h"
#include "db.h"
#include "stock.h"

#define BUEVENTVIEW_TABLE_UNCHANGED	(char *)0
#define BUEVENTVIEW_TABLE_MODIFIED	(char *)1
#define BUEVENTVIEW_TABLE_BIGGER	(char *)2
#define BUEVENTVIEW_TABLE_SMALLER	(char *)3

#define BUEVENTVIEW_THREAD_STATE_RUN	0
#define BUEVENTVIEW_THREAD_STATE_PAUSE	1
#define BUEVENTVIEW_THREAD_STATE_STOP	2

#define BUEVENTVIEW_SORT_EVENTS_ORIGINAL	0
#define BUEVENTVIEW_SORT_EVENTS_LDDATE		1
#define BUEVENTVIEW_SORT_EVENTS_LDDATE_VIEW	2

typedef int (BU_EventViewCallbackProc) (int type, BU_ClientData clientData, void *data);

typedef struct BUEV_Table {
	char name[32];
	Dbptr dbtable;
	long nrows;
	char fname[512];
	double mod_time;
} BUEV_Table;

typedef struct BUEV {
	char *dbname;
	Dbptr db;
	BUEV_Table event;
	BUEV_Table origin;
	BUEV_Table origerr;
	BUEV_Table assoc;
	BUEV_Table arrival;
	BUEV_Table wfmeas;
	BUEV_Table netmag;
	BUEV_Table stamag;
	BUEV_Table mt;
	BUEV_Table site;
	BUEV_Table detection;

	Tbl *kpat_evid_match;
	Tbl *tpat_evid_match;
	Hook *hook_evid_match;
	Tbl *kpat_orid_match;
	Tbl *tpat_orid_match;
	Hook *hook_orid_match;
	Tbl *kpat_event_origin_match;
	Tbl *tpat_event_origin_match;
	Hook *hook_event_origin_match;
	Tbl *kpat_origin_origerr_match;
	Tbl *tpat_origin_origerr_match;
	Hook *hook_origin_origerr_match;
	Tbl *kpat_origin_netmag_match;
	Tbl *tpat_origin_netmag_match;
	Hook *hook_origin_netmag_match;
	Tbl *kpat_origin_mt_match;
	Tbl *tpat_origin_mt_match;
	Hook *hook_origin_mt_match;
	Tbl *kpat_origin_assoc_match;
	Tbl *tpat_origin_assoc_match;
	Hook *hook_origin_assoc_match;
	Tbl *kpat_assoc_arrival_match;
	Tbl *tpat_assoc_arrival_match;
	Hook *hook_assoc_arrival_match;
	Hook *hook_site_match;
	Tbl *kpat_netmag_stamag_match;
	Tbl *tpat_netmag_stamag_match;
	Hook *hook_netmag_stamag_match;
	Tbl *kpat_stamag_wfmeas_match;
	Tbl *tpat_stamag_wfmeas_match;
	Hook *hook_stamag_wfmeas_match;
	Tbl *kpat_origin_detection_match;
	Tbl *tpat_origin_detection_match;
	Hook *hook_origin_detection_match;
	Tbl *kpat_detection_match;
	Tbl *tpat_detection_match;
	Hook *hook_detection_match;

	BU_EventViewCallbackProc *monitor_callback;
	BU_ClientData monitor_clientdata;
	double monitor_time_increment;
	pthread_t monitor_thread;
	int monitor_state;
} BUEV;

typedef struct BUEVMoment {
	long mtid;
	double lddate;
	double m[6];
	Dbptr dbmt;
} BUEVMoment;

typedef struct BUEVStamag {
	long magid;
	double magnitude;
	Dbptr dbstamag;
	Dbptr dbwfmeas;
} BUEVStamag;

typedef struct BUEVMagnitude {
	long magid;
	char magtype[8];
	double magnitude;
	Dbptr dbnetmag;
	Tbl *stamags;
} BUEVMagnitude;

typedef struct BUEVDetectionStachan {
	char sta[16];
	char chan[16];
	double lat;
	double lon;
	double elev;
	double distance;
	double predicted_P_time;
	double predicted_S_time;
	Dbptr dbsite;
} BUEVDetectionStachan;

typedef struct BUEVDetection {
	double time;
	double residual;
	double lddate;
	char phase[16];
	Dbptr dbdetection;
	BUEVDetectionStachan station;
} BUEVDetection;

typedef struct BUEVOrigin {
	long orid;
	long ndef;
	long nass;
	double time;
	double lat;
	double lon;
	double depth;
	double lddate;
	int preferred;
	Dbptr dborigin;
	Dbptr dborigerr;
	Arr *stations;
	Tbl *associations;
	Tbl *detections;
	Tbl *magnitudes;
	Tbl *moments;
	BUEVMagnitude *pref_magnitude;
	BUEVMoment *pref_moment;
} BUEVOrigin;

typedef struct BUEVOriginStation {
	BUEVOrigin *origin;
	char sta[16];
	double lat;
	double lon;
	double elev;
	double distance;
	double predicted_P_time;
	double predicted_S_time;
	Dbptr dbsite;
} BUEVOriginStation;

#define BUEVENTVIEW_ASSOCIATION_DEFINING_DEFINING	0
#define BUEVENTVIEW_ASSOCIATION_DEFINING_ASSOCIATED	1
#define BUEVENTVIEW_ASSOCIATION_DEFINING_OTHER		2

typedef struct BUEVAssociation {
	long arid;
	int defining;
	Dbptr dbassoc;
	Dbptr dbarrival;
	BUEVOriginStation *station;
} BUEVAssociation;

typedef struct BUEVEvent {
	long evid;
	long prefor;
	long order;
	double lddate;
	double lddate_view;
	Dbptr dbevent;
	Tbl *origins;
	BUEVOrigin *pref_origin;
	BUEVMagnitude *pref_magnitude;
	BUEVMoment *pref_moment;
} BUEVEvent;

#ifdef  __cplusplus

class BUEventView : public BUConfigure {
public:
	BUEventView (Pf *params=NULL);
	~BUEventView(void);
	int reset();
	int refreshEvents (Tbl **events);
	int refreshEvent (long dbevent_record, BUEVEvent *event, long *order=NULL);
	int refreshOrigin (long dborigin_recno, BUEVOrigin *origin);
	int refreshAssociations (BUEVOrigin *origin);
	int refreshAssociations (BUEVEvent *event);
	int refreshAssociation (BUEVOrigin *origin, long dbassoc_record, BUEVAssociation *association);
	int refreshDetections (Tbl **detections, double t1, double t2);
	int refreshDetections (BUEVOrigin *origin, int all=0);
	int refreshStamags (BUEVMagnitude *magnitude);
	int refreshStamag (long dbstamag_record, BUEVStamag *stamag);
	BUEVEvent *findEvid (Tbl *events, long evid);
	BUEVEvent *findEvid (long evid);
	BUEVOrigin *findOrid (BUEVEvent *event, long orid);
	BUEVOrigin *findOrid (long orid);
	void freeEvents (Tbl *events);
	void freeEvent (BUEVEvent *event);
	void freeOrigins (Tbl *origins);
	void freeOrigin (BUEVOrigin *origin);
	void freeStations (Arr *stations);
	void freeAssociations (Tbl *associations);
	void freeDetections (Tbl *detections);
	void freeMagnitudes (Tbl *magnitudes);
	void freeMoments (Tbl *moments);
	void freeStamags (Tbl *stamags);
	void numbers (long *events, long *origins, long *detections, long *arrivals, long *assocs, long *mts, long *magnitudes);
	int checkChanges (Arr **changes);
	int monitorChangesThread (double time_increment, BU_EventViewCallbackProc *callbackProc, BU_ClientData clientData);
	int sortEvents (Tbl *events, long sort_type);
	BUEV buev;
private:
	int init(void);
	int lookupStation (BUEVOrigin *origin, char *sta, double statime, BUEVOriginStation **station);
	int lookupStation (BUEVDetection *detection, char *sta, BUEVOrigin *origin = NULL);
};

#endif

#endif
