/* Copyright (c) 1997-2004 Boulder Real Time Technologies, Inc. */
/* All rights reserved */

/* This software module is wholly owned by Boulder Real Time 
   Technologies, Inc. Any use of this software module without
   express written permission from Boulder Real Time Technologies,
   Inc. is prohibited. */

#ifndef _brttpkt_
#define _brttpkt_

#include <pthread.h>

#include "brttutil.h"
#include "Pkt.h"
#include "orb.h"
#include "db.h"
#include "response.h"

#ifdef  __cplusplus
extern "C" {
#endif

#define	BRTTPKT_PIPE_INCON	-2
#define	BRTTPKT_PIPE_OUTOFRANGE	-1
#define	BRTTPKT_PIPE_NOTCONTIG	0
#define	BRTTPKT_PIPE_CONTIG	1
#define	BRTTPKT_PIPE_DUP	2
#define	BRTTPKT_PIPE_OVERLAP	3

#define	BRTTPKT_AVE_RMS		1
#define	BRTTPKT_AVE_MS		2
#define	BRTTPKT_AVE_MEAN	3

#define	BRTTPKT_CONTIG_TOLERANCE	(0.5)	/* Packet contiguity tolerance in dts */

#define	PKTCHANNELPIPE_FIRST	1
#define	PKTCHANNELPIPE_CONTIG	2
#define	PKTCHANNELPIPE_DUP	3
#define	PKTCHANNELPIPE_OVERLAP	4
#define	PKTCHANNELPIPE_GAP	5
#define	PKTCHANNELPIPE_EARLY	6

#define	PKTCHANNELCALIB_LAT		(0x1)
#define	PKTCHANNELCALIB_LON		(0x2)
#define	PKTCHANNELCALIB_ELEV		(0x4)
#define	PKTCHANNELCALIB_CALIB		(0x8)
#define	PKTCHANNELCALIB_CALPER		(0x10)
#define	PKTCHANNELCALIB_SEGTYPE		(0x20)
#define	PKTCHANNELCALIB_RESPONSE	(0x40)

#define ORBREAPTHR_OK			(1)
#define ORBREAPTHR_NODATA		(2)
#define ORBREAPTHR_TIMEOUT		(3)
#define ORBREAPTHR_STOPPED		(4)
#define ORBREAPTHR_NEWDATA		(5)

#define	CMDTHREAD_COMPLAIN	1
#define	CMDTHREAD_EXEC		2
#define	CMDTHREAD_POSTING	3

#define	CONVERT_WF_NONE		0
#define	CONVERT_WF_FH2IH	1
#define	CONVERT_WF_IH2FH	2
#define	CONVERT_WF_F42S4	3
#define	CONVERT_WF_T42I4	4
#define	CONVERT_WF_I42T4	5
#define	CONVERT_WF_S42F4	6
#define	CONVERT_WF_I42S4	7
#define	CONVERT_WF_S42I4	8
#define	CONVERT_WF_T42F4	9
#define	CONVERT_WF_F42T4	10
#define	CONVERT_WF_IH		11
#define	CONVERT_WF_FH		12
#define	CONVERT_WF_S4		13
#define	CONVERT_WF_I4		14
#define	CONVERT_WF_T4		15
#define	CONVERT_WF_F4		16

typedef struct pktchan_ {
	double time;
	double dt;
	int contig;
	unsigned int pktid;
	int nsamp;
	int datasz;
	float *data;
} Pktchan;

typedef struct pktchanpipe_ {
	double lasttime;
	char *netstachan;
	int maxpkts;
	Tbl *pktchans;
	Pktchan lastpkt;
} Pktchanpipe;

typedef struct pktchanpipeproc_ {
	long npp;
	Arr *pps_arr;
	Arr *ippc_arr;
	Tbl *pps_tbl;
	Tbl *nscs_tbl;
	int *process;
} Pktchanpipeproc;

typedef struct pktchanave_ {
	double time;
	double nexttime;
	double twin;
	double tpad;
	double tmin;
	double tmingap;
	double tsnr;
	double tsnr_noise;
	double hold_time;
	double hold_nsum;
	double hold_sum;
	float rc_f;
	float rc_r;
	float rc_f_noise;
	float rc_r_noise;
	int hold_lasti;
	int hold_lastj;
	int type;
	int nsum_size;
	double *nsum;
	int npktchans;
	int pktchans_size;
	Pktchan **pktchans;
} Pktchanave;

typedef struct pktchanproc_ {
	double tstart;
	double tend;
	int npktchans;
	int pktchans_size;
	Pktchan *pktchans;
	void *pvt;
	int (*callback) ();
} Pktchanproc;

typedef struct pktchanbuf_ {
	double twin;
	double time;
	double endtime;
	double lastpkttime;
	double dt;
	long bufsamp;
	int sizetimes;
	int lastpktoff;
	int nextpktoff;
	int ntimes;
	float *buf;
	double *times;
	int *itimes;
} Pktchanbuf;

typedef struct chantracebuf_ {
	double tstart;
	double tend;
	double latency;
	double time;
	double dt;
	double calib;
	double calper;
	int apply_calib;
	int nsamp;
	int datasz;
	int ngood;
	float *data;
	char segtype[16];
} Chantracebuf;

typedef struct chantraceproc_spec_ {
	char *net_expr;
	Hook *net_hook;
	char *sta_expr;
	Hook *sta_hook;
	char *chan_expr;
	Hook *chan_hook;
	int (*callback) ();
	void *pvt;
	char *filter;
	double tstart;
	double tend;
	double latency;
	int apply_calib;
} CTPS;

typedef struct chantraceproc_buf_ {
	CTPS *spec;
	Chantracebuf *buf;
} CTPB;

typedef struct chantraceproc_ {
	Tbl *chanspecs;
	Arr *chanbufs;
	Arr *nomatch;
	double latency;
	double maxwaittime;
	void *reap;
} Chantraceproc;

typedef struct pktchannelpipe_ {
	Arr *select_arr;
	Tbl *reject_tbl;
	Arr *channels_select;
	Arr *channels_reject;
	int maxpkts;
	int (*callback) (void *pvt, PktChannel *pktchan, int queue_code, double gaptime);
	void *pvt;
	Packet *pkt;
} PktChannelPipe;

typedef struct pktchancalib_ {
	Arr *channels_arr;
	Arr *stations_arr;
	char *dbname;
	double mod_time;
	char *dbpath;
	Dbptr db;
	int need_calib;
	int need_response;
	int need_site;
} PktChannelCalib;

typedef struct pktchannel2trace_ {
	Dbptr dbm;
	Dbptr dbt;
	Dbptr dbw;
	int need_calib;
	int need_response;
	int need_site;
	int regular_timing;
	int pktchanfloat_flag;
	double tstart;
	double samprate;
	int nsamp;
	Tbl *wf_matches;
	Hook *calib_hook;
	Hook *wf_hook;
} PktChannel2Trace ;

typedef struct orbreapthr_ {
	Pmtfifo *mtfifo;
	pthread_t run_thread;
	pthread_mutex_t mutex;
	pthread_mutex_t mutex_restart_reap;
	char *orbname;
	char *select;
	char *reject;
	double after;
	int orb;
	int block;
	int halt;
	int pause;
	int reaping;
	int started;
	Tbl *callbacks;
} OrbreapThr;

typedef struct orbputthr_ {
	Pmtfifo *mtfifo;
	pthread_t run_thread;
	int orb;
	int halt;
	int putting;
} OrbputThr;

typedef struct cmdthread_ {
	char *target;
	char *type;
	char *username;
	char *hostname;
	char *ip;
	Pf *pf_auth;
	int orbcmd;
	int orbrsp;
	void *user;
	pthread_t reap_thread;
	pthread_t keepalive_thread;
	int reaping;
	int halt;
	char *cmd;
	char *cmd_username;
	char *cmd_hostname;
	char *cmd_ip;
	char *cmd_dlnames;
	char *sequence_target;
	int cmd_count;
	OrbputThr *put_thr;
	int (*callback) (struct cmdthread_ *cth, void *user, int what, ...);
} CmdThread;

typedef struct pktchannel2buffer_chan_ {
	char snet[8];
	char ssta[16];
	char schan[16];
	char sloc[8];
	double samprate;
	double calib;
	double calper;
	char segtype[16];
	char datatype[4];
	double time;
	int nsamp;
	int isfloat;
	size_t data_size;
	int *data;
	size_t heap;
	char why[10];
} PktChannel2BufferChan;

typedef struct pktchannel2buffer_ {
	Arr *channels_arr;
	int (*flush_callback) (PktChannel2BufferChan *bchan, void *user_data);
	void *flush_callback_user_data;
	pthread_mutex_t mutex;
} PktChannel2Buffer;

typedef struct pktchannel2db_ {
	PktChannel2Buffer *pbuf;
	Arr *ignore_arr;
	Dbptr dbwfdisc;
	double time_write_wf;
	double time_new_wfdisc;
	double time_update_wf;
	double time_dbmatches;
	size_t bytes_flushed;
	int max_open_fds;
	int verbose;
	int add_wfdisc_headers;
	int update_lddate;
	char datatype[4];
	char datatype_param[4];
	double gain;
	int convert_type;
	float fgap_value;
	double wfduration;
	int igap_value;
	int addcheck;
	size_t maxheap;
	double maxduration;
	double realtime_thresh;
	double samprate_thresh;
	double calib_thresh;
	double calper_thresh;
	Tbl *matches_pattern;
	Hook *matches_hook;
	Hook *addchk_hook;
	char wfname_pattern[256];
} PktChannel2Db;

typedef struct pktrepackage_chan_segment_ {
	double pkttime;
	double time;
	double dt;
	int nsamp;
	int data_sz;
	int *data;
	int isfloat;
} PktRepackageChanSegment;

typedef struct pktrepackage_chan_ {
	double nexttime;
	double dt;
	double calib;
	double calper;
	char net[8];
	char sta[16];
	char chan[16];
	char loc[8];
	char segtype[16];
	Tbl *segments;
} PktRepackageChan;

typedef struct pktrepackage_ {
	Packet *pkt;
	char *suffix;
	char *subcode;
	int orb;
	double twin;
	double time;
	double endtime;
	Arr *channels_arr;
	Tbl *channels_tbl;
	int (*flush_callback) (void *user_data);
	void *flush_callback_user_data;
	pthread_mutex_t mutex;
} PktRepackage;

extern Chantracebuf * chantracebuf_new ( double tstart, double tend, double latency, int apply_calib );
extern int chantracebuf_free ( Chantracebuf *cp );
extern int chantracebuf_add ( Chantracebuf *cp, PktChannel *pktchan );
extern int chantracebuf_filter ( void **fil, int *type, Chantracebuf *icp, Chantracebuf **ocp, char *filspec, int filinit );
extern Chantraceproc * chantraceproc_new ( double latency, double maxwaittime );
extern int chantraceproc_free ( Chantraceproc *cp );
extern CTPS * chantraceproc_spec_new ( void );
extern void chantraceproc_spec_free ( void *ptr );
extern CTPB * chantraceproc_buf_new ( void );
extern void chantraceproc_buf_free ( void *ptr );
extern int chantraceproc_addchan ( Chantraceproc *cp, char *netstachanp, int (*callback)(), void *pvt, double tstart, double tend, double latency, int apply_calib );
extern int chantraceproc_process ( Chantraceproc *cp, int orb );
extern CTPS * chantraceproc_match ( Chantraceproc *cp, char *net, char *sta, char *chan );
extern Pktchanave * pktchanave_new ( double twin, double tpad, double tmin, double tmingap, char *type );
extern int pktchanave_proc ( Pktchanave *pa, Pktchan *pktchan, Pktchan *pktchana );
extern int pktchanave_snr ( Pktchanave *pa, double time, double endtime, double tsnr_signal, double tsnr_noise, double time_noise_floor, double endtime_noise_floor, int *npkts, Pktchan **pkts, int *pktsize );
extern Pktchanbuf * new_pktchanbuf ( double twin );
extern int pktchanbuf_add ( Pktchanbuf *pb, Pktchan *pktchan );
extern Pktchanpipeproc * new_pktchanpipe ( void );
extern int pktchanpipe_add ( Pktchanpipeproc *ppc, char * netstachan, int maxpkts );
extern int pktchanpipe_push ( Pktchanpipeproc * ppc, char * netstachan, PktChannel * pktchannel );
extern int pktchanpipe_pop ( Pktchanpipeproc *ppc, char ** netstachan, Pktchan ** pktchan, int * contig );
extern Pktchanproc * pktchanproc_new ( int (*callback)(), void * pvt );
extern int pktchanproc_settwin ( Pktchanproc * pp, double tstart, double tend );
extern int pktchanproc_push ( Pktchanproc * pp, Pktchan * pktchan );
extern PktChannelPipe *pktchannelpipe_new (Tbl *channels_select, Tbl *channels_reject, int maxpkts, 
			int (*callback) (void *pvt, PktChannel *pktchan, int queue_code, double gaptime), 
			void *pvt);
extern int pktchannelpipe_free (PktChannelPipe *pcp);
extern int pktchannelpipe_push (PktChannelPipe *pcp, char *srcname,
			double pkttime, char *pkt, int nbytes);
extern int pktchannelpipe_flush (PktChannelPipe *pcp);

extern PktChannelCalib *pktchannelcalib_new (char *dbname, 
			int need_calib, int need_response, int need_site);
extern int pktchannelcalib_free (PktChannelCalib *ch);
extern int pktchannelcalib_get (PktChannelCalib *ch, 
			char *snet, char *ssta, char *schan, char *sloc, double time, int check,
			char *sta, char *chan, double *lat, double *lon, double *elev,
			double *calib, double *calper, char *segtype, Response **response);
extern PktChannel2Trace *pktchannel2trace_new (Dbptr dbm, Dbptr dbt, Dbptr dbwf, int need_calib, int need_response, int need_site);
extern int pktchannel2trace_free (PktChannel2Trace *pt);
extern int pktchannel2trace_destroy (PktChannel2Trace *pt);
extern int pktchannel2trace_put (PktChannel2Trace *pt, PktChannel *pchan, int calibfromdb, double tstart, double tend);
extern int pktchannel2trace_put_from_db (PktChannel2Trace *pt, char *sta, char *chan_expr, int calibfromdb, double tstart, double tend);
extern int pktchannel2trace_get_stats (PktChannel2Trace *pt, char *sta, char *chan, int *ngood, int *nbad);
extern PktChannel *pktchannelcopy (PktChannel *pktchan);
extern OrbreapThr *orbreapthr_new (int orb, double timeout, int queuesize);
extern OrbreapThr *orbreapthr_new2 (char *orbname, char *select, char *reject, double tafter, double timeout, int queuesize);
extern int orbreapthr_register_callback (OrbreapThr *orbth, void (*callback) (int reason, OrbreapThr *orbthr, void *user_data), void *user_data);
extern int orbreapthr_pause (OrbreapThr *prbthr, int flush);
extern int orbreapthr_continue (OrbreapThr *prbthr);
extern int orbreapthr_select (OrbreapThr *prbthr, char *select);
extern int orbreapthr_reject (OrbreapThr *orbthr, char *reject);
extern int orbreapthr_after (OrbreapThr *orbthr, double after);
extern int orbreapthr_seek (OrbreapThr *orbthr, int which);
extern int orbreapthr_set_to_stop (OrbreapThr *orbth);
extern int orbreapthr_is_stopped (OrbreapThr *orbth);
extern int orbreapthr_stop_and_wait(OrbreapThr *orbth);
extern int orbreapthr_destroy (OrbreapThr *orbth);
extern int orbreapthr_set_timeout (OrbreapThr *orbth, double timeout);
extern int orbreapthr_data_available (OrbreapThr *orbth);
extern int orbreapthr_get (OrbreapThr *orbth, int *pktid, char *srcname, 
				double *time, char **packet, int *nbytes, int *bufsize);
extern int orbreapthr_set_verbose (int v);
extern OrbputThr *orbputthr_new (int orb, int queuesize);
extern int orbputthr_stop (OrbputThr *orbth);
extern int orbputthr_destroy (OrbputThr *orbth);
extern int orbputthr_put ( OrbputThr *orbth, char *srcname, double time, char *packet, int nbytes );
extern int initcmd (char *cmd, char *reply, char *targets, char *types,
			char *dlnames, double delayhangup, char *sequence, int sequence_sz, void **pfbuf_vstack );
extern int pfcmdstring2orb ( char *pfstring, char *name, int orb );
extern int orbpkt2cmdpf ( char *srcname, char *packet, int nbytes, char *target, Pf **pfcmd, int *dsig_ok );
extern int initrsp (char *cmd, char *target, char *type, char *dlname, char *dlserial, char *dlcmd, Pf *pfcmd, int cmdcnt,
		char **username, char **hostname, char **ip, char **sequence_target,
		char *disposition, char *srcname, void **pfbuf_vstack);
extern CmdThread *cmdthread_new (int orbcmd, int orbrsp, char *target,
		char *type, int (*callback) (CmdThread *cth, void *user, int what, ...), void *user);
extern int cmdthread_destroy (CmdThread *cth);
extern int cmdthread_send_rsp (CmdThread *cth, Pf *pfcmd, char *dlname, char *dlserial, char *dlcmd, Pf *pfrsp, char *disposition);
extern PktChannel2Buffer *pktchannel2buffer_create(int (*flush_callback) (PktChannel2BufferChan *bchan, void *user_data), void *flush_callback_user_data);
extern void pktchannel2buffer_free (PktChannel2Buffer *pbuf);
extern PktChannel2BufferChan * pktchannel2buffer_chan_copy (PktChannel2BufferChan *cbuf);
extern void pktchannel2buffer_chan_free (PktChannel2BufferChan *cbuf);
extern int pktchannel2buffer_put (PktChannel2Buffer *buf, PktChannel *pktchan, long maxheap, double maxduration, int verbose);
extern int pktchannel2buffer_flush (PktChannel2Buffer *buf); 
extern PktChannel2Db *pktchannel2db_create(Dbptr dbwfdisc, Pf *params);
extern void pktchannel2db_free (PktChannel2Db *pdb);
extern int pktchannel2db_put (PktChannel2Db *pdb, PktChannel *chan);
extern void pktchannel2db_convert_wf (int convert_type, int *buf, long nsamp, float fgap_value, int igap_value);
extern int pktchannel2db_redb (PktChannel2Db *pdb, Dbptr db);
extern int pktchannel2wfdisc (PktChannel *pktchan, char *wfname_template, int convert, double maximum_flagged_gap_duration, Dbptr dbout);
extern PktRepackage *pktrepackage_create (double twin, char *suffix, char *subcode, int orb, 
				int (*flush_callback) (void *user_data), void *flush_callback_user_data);
extern int pktrepackage_put (PktRepackage *buf, char *net, char *sta, char *chan, char *loc,
				double time, double dt, double calib,
				double calper, char *segtype, int nsamp, int *data, int isfloat);
extern int pktrepackage_put_pktchan (PktRepackage *buf, PktChannel *pktchan);
extern int pktrepackage_flush (PktRepackage *buf);

#ifdef	__cplusplus
}
#endif

#endif
